package cologne.eck.all_peas.control;


/*
 * Peafactory - Production of Password Encryption Archives
 * Copyright (C) 2015  Axel von dem Bruch
 * 
 * This library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published 
 * by the Free Software Foundation; either version 2 of the License, 
 * or (at your option) any later version.
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE. 
 * See the GNU General Public License for more details.
 * See:  http://www.gnu.org/licenses/gpl-2.0.html
 * You should have received a copy of the GNU General Public License 
 * along with this library.
 */

/**
 * Parent class of all pea frames except image pea. 
 */


import java.io.File;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;

import org.bouncycastle.crypto.BlockCipher;
import org.bouncycastle.crypto.Digest;

import cologne.eck.all_peas.cloud.CloudControl;
import cologne.eck.all_peas.cloud.CloudPropertyHandler;
import cologne.eck.all_peas.cloud.gui.UploadWorker;
import cologne.eck.all_peas.data.AttachmentHandler;
import cologne.eck.all_peas.data.PeaProperties;
import cologne.eck.all_peas.vm_specific.JREProperties;
import cologne.eck.contact_pea.gui.LockFrameContact;
import cologne.eck.peafactory.crypto.AlgoParamHandler;
import cologne.eck.peafactory.crypto.AuthenticatedEncryption;
import cologne.eck.peafactory.crypto.CipherStuff;
import cologne.eck.peafactory.crypto.HashStuff;
import cologne.eck.tools.ReadResources;
import cologne.eck.tools.TestLog;
import cologne.eck.tools.UnexpectedValueException;
import cologne.eck.tools.WriteResources;




/**
 * Content related utilities. 
 * 
 * @author Axel von dem Bruch
 *
 */
public abstract class ContentUtilities {
	
	/**
	 * Error message from last decryption process
	 */
	private static String decryptErrorMessage = null;
	
	/**
	 * Decrypt a file and return the plain text as byte array. 
	 * The session key must have been set before. 
	 * If an error occurred,an error message is set in form of 
	 * fileName - errorMessage and can be accessed by getDecryptErrorMessage()
	 * 
	 * @param fileName	file name with absolute path
	 * 
	 * @return	plain text as byte array or null: see getDecrytErrorMessage() 
	 */
	public final static byte[] decryptFile(String fileName) {
		// reset:
		decryptErrorMessage = null;
		byte[] cipherText = ReadResources.readExternFile(fileName);
		if (cipherText == null) {
			TestLog.ve(ContentUtilities.class, "Can't read file: " + fileName);
			decryptErrorMessage = fileName + " - " + PeaProperties.getVmBridge().translate("no_access");
			return null;
		}
		// check password identifier
		byte[] keyMaterial = CipherStuff.getSessionKeyCrypt().getKey()[0];
		String error = PeaControl.checkPassword(cipherText, keyMaterial);
		if (error != null) {
			TestLog.ve(ContentUtilities.class, "Error in key derivation: " + fileName);
			decryptErrorMessage = fileName + " - " + error;
			return null;
		} 
		// decrypt
		byte[] plainText = CipherStuff.getInstance().decrypt(
				cipherText, keyMaterial, true);
		if (plainText == null) {
			//new UnexpectedValueException("byte[]", "plainText", "is null").printDescription();
			TestLog.e(LockFrameContact.class, CipherStuff.getErrorMessage());
			decryptErrorMessage = fileName + " - " + PeaProperties.getVmBridge().translate("failed")
					+ ", " + CipherStuff.getErrorMessage();
			return null;
		}
		return plainText;
	}	
	
	/**
	 * Encrypt and store the content in an existing file. 
	 * The key is then stored encrypted as session key. 
	 * 
	 * @param keyMaterial	the key to encrypt: must be null if cipherBytes are not null
	 * @param plainBytes	the content to encrypt 
	 * @param fileName		the existing file to store the encrypted content with absolute path
	 * 
	 * @return	true if content was saved successfully
	 */
	public final static boolean saveContent(
			byte[] keyMaterial, byte[] plainBytes, 
			String fileName) {

		if (new File(fileName).exists() == false) {
			TestLog.e(ContentUtilities.class, "ContentUtilities save: file does not exist: " + fileName);
			new UnexpectedValueException("String", "fileName", "does not eist: " + fileName).printDescription();
			return false;
		}
		if (new File(fileName).isFile() == false) {
			TestLog.e(ContentUtilities.class, "ContentUtilities save: not file " + fileName);
			new UnexpectedValueException("String", "fileName", "is not a name of a file: " + fileName).printDescription();
			return false;
		}
		if (plainBytes == null) { // && cipherBytes == null) {
			TestLog.e(ContentUtilities.class, "ContentUtilities save: Plaintext is null: " + fileName);
			new UnexpectedValueException("byte[]", "plainBytes", "null for file: " + fileName).printDescription();
			return false;
		}
		byte[] cipherBytes = CipherStuff.getInstance().encrypt(plainBytes, 
				keyMaterial, // keyMaterial is overwritten
				true, null ); // key is stored encrypted
		if (cipherBytes == null) {
			TestLog.e(ContentUtilities.class, "Decryption error - " + CipherStuff.getErrorMessage());
			TestLog.ve(ContentUtilities.class, "Missing ciphertext", 5);
			return false;
		}
		boolean success = WriteResources.write(cipherBytes,fileName, null);
		if (success == false) {
			TestLog.ve(ContentUtilities.class, "Write access failed: " + fileName);
		}		
		if (fileName.contains(CloudControl.getTmpDir())) {
			// save in cloud:
			ContentUtilities.saveTmpFileInCloud(fileName);
		}
		return success;
	}
	
	/**
	 * Save one single backup file of a cloud file
	 * that is stored in temporary directory
	 * 
	 * @param tmpFileName	the name of the backup file in tmp folder to store
	 */
	public static void saveTmpFileInCloud(String tmpFileName) {
		if (tmpFileName.contains(CloudControl.getTmpDir())) {
			// save in cloud:
			String providerName = CloudControl.getProviderNameFromTmpFile(tmpFileName);
			File[] fileArray = {new File(tmpFileName)};
			TestLog.v(ContentUtilities.class, "Upload " + tmpFileName + "... ");
			new UploadWorker(JREProperties.getMainWindow(), 
					providerName, 
					null, 
					fileArray, 
					null, false).execute();			
		} else {
			TestLog.e(ContentUtilities.class, "ContentUtilities saveTmpFileInCloud: " + tmpFileName
					+ " - is not a cloud file backup");
		}
	}
	
	/**
	 * 
	 * @param fileNames			file names to re-encrypt
	 * @param previousKeyAndAlgos	old key and algos to decrypt
	 * @param newKeyAndAlgos		new key and algos to encrypt
	 * @param reencryptOnDisk	true: decrypt large files on disk, then encrypt due
	 * 		to heap error (file.lentgh > 256 MB) - check file length before
	 * 		and warn user, that files are unencrypted on disk for a while
	 * 
	 * @return	null for success, a localized error message to show otherwise
	 */
	public static final String reencrypt(String[] fileNames, 
			byte[][] previousKeyAndAlgos,
			byte[][] newKeyAndAlgos,
			
			//byte[] oldKey, byte[] newKey,
			//BlockCipher oldCipher, BlockCipher newCipher, Digest oldHash, Digest newHash, 
			boolean reencryptOnDisk) {
		if (fileNames == null) {
			TestLog.e(ContentUtilities.class, "No file to re-encrypt");
			TestLog.ve(ContentUtilities.class, "file names to re-encrypt is null", 5);
			return PeaProperties.getVmBridge().translate("file_not_found");
		}		
		if (previousKeyAndAlgos == null) {
			TestLog.e(ContentUtilities.class, "No key to decrypt");
			TestLog.ve(ContentUtilities.class, "old key is null", 5);
			return PeaProperties.getVmBridge().translate("unexpected_error") + " (missing old key)";
		}
		if (newKeyAndAlgos == null) {
			TestLog.e(ContentUtilities.class, "No  key to encrypt");
			TestLog.ve(ContentUtilities.class, "new key is null", 5);
			return PeaProperties.getVmBridge().translate("unexpected_error") + " (missing new key)";
		}
		if (Arrays.equals(previousKeyAndAlgos[0], newKeyAndAlgos[0])) {
			TestLog.e(ContentUtilities.class, "Re-encryption key and previous key are equal");
			TestLog.ve(ContentUtilities.class, "same keys", 5);
			return null;//"re-encryption with same key, break re-encryption...";
		}
		BlockCipher oldCipher = AlgoParamHandler.getCipherFromString(
				new String(previousKeyAndAlgos[1], AttachmentHandler.getASCIICharset()));
		Digest oldHash = AlgoParamHandler.getHashFromString(
				new String(previousKeyAndAlgos[1], AttachmentHandler.getASCIICharset()));
		
		BlockCipher newCipher = AlgoParamHandler.getCipherFromString(
				new String(newKeyAndAlgos[1], AttachmentHandler.getASCIICharset()));
		Digest newHash = AlgoParamHandler.getHashFromString(
				new String(newKeyAndAlgos[1], AttachmentHandler.getASCIICharset()));

		BlockCipher currentCipher = CipherStuff.getCipherAlgo(); // cipher to reset
		StringBuilder errorBuilder = new StringBuilder();
		AuthenticatedEncryption ae = CipherStuff.getCipherMode();
		for (String fileNameWithPath : fileNames) {
			if (fileNameWithPath == null) {
				errorBuilder.append(PeaProperties.getVmBridge().translate("unexpected_error") + " - file name is null \n");
				continue;
			}
			String fileName = null;
			String dirName = null;
			try {
				if (Paths.get(fileNameWithPath).getFileName() != null){
					Path filePath = Paths.get(fileNameWithPath).getFileName();
					if (filePath != null) {
						fileName = filePath.toString();
					} else {
						TestLog.ve(ContentUtilities.class, "Missing Path", 5);
						errorBuilder.append(PeaProperties.getVmBridge().translate("unexpected_error") + "  \n");
						continue;
					}
				} else {
					TestLog.e(ContentUtilities.class, "Can't get file name: " + fileName);
					errorBuilder.append(PeaProperties.getVmBridge().translate("unexpected_error") + " - file name is null \n");
					continue;
				}
				if (Paths.get(fileNameWithPath).getParent() != null){
					Path dirPath = Paths.get(fileNameWithPath).getParent();
					if (dirPath != null) {
						dirName = dirPath.toString();
					} else {
						TestLog.ve(ContentUtilities.class, "Missing Path", 5);
						errorBuilder.append(PeaProperties.getVmBridge().translate("unexpected_error") + "  \n");
						continue;
					}
				}
			} catch (Exception e) {
				errorBuilder.append(PeaProperties.getVmBridge().translate("unexpected_error") + " - file name is invalid: "+ fileNameWithPath + "\n");
				continue;
			}

			long fileLen = new File(fileNameWithPath).length();
			if (fileLen >= 1024 * 1024 * 256) { // > 256 MB a heap error may occur
				if (reencryptOnDisk == true) {
					CipherStuff.setCipherAlgo(oldCipher);
					String[] names = {fileNameWithPath};					
					String[] decryptError = ae.decryptFiles(names, 
							Arrays.copyOf(previousKeyAndAlgos[0], previousKeyAndAlgos[0].length), false, null);

					if (decryptError[0] != null) {
						if (decryptError[0].equals(PeaProperties.getVmBridge().translate("data_integrity_violated"))){					
							TestLog.e(ContentUtilities.class, PeaProperties.getVmBridge().translate("data_integrity_violated")
									+ ": " + fileNameWithPath + "\n"
									+ PeaProperties.getVmBridge().translate("integrity_check_failed_message"));
							// The file was decrypted
							return 
									fileNameWithPath + ":\n" + PeaProperties.getVmBridge().translate("data_integrity_violated")
									+"\n" + PeaProperties.getVmBridge().translate("integrity_check_failed_message")
									+ "\n\n" + PeaProperties.getVmBridge().translate("warning") + ":\n " 
									+ fileNameWithPath + " - " + PeaProperties.getVmBridge().translate("unencrypted") + "\n"
									+ PeaProperties.getVmBridge().translate("open_file") + " " + fileNameWithPath;
						}
					}
					CipherStuff.setCipherAlgo(newCipher);
					String[] encryptError = ae.encryptFiles(names, 
							Arrays.copyOf(newKeyAndAlgos[0], newKeyAndAlgos[0].length), false, null, null);
					if (encryptError[0] != null) {
						errorBuilder.append( encryptError[0] + ": " + fileNameWithPath);
						continue;
					}
				} else {
					errorBuilder.append(PeaProperties.getVmBridge().translate("file_too_large") 
							+ fileNameWithPath + "\n");
					continue;
				}
				continue;
			}
			byte[] oldEncrypted = ReadResources.readExternFile(fileNameWithPath);
			if (oldEncrypted == null) {
				TestLog.e(ContentUtilities.class, "Read failed: " + fileNameWithPath);
				TestLog.ve(ContentUtilities.class, ReadResources.getLastErrorMessage(), 5);
				errorBuilder.append(//PeaProperties.getVmBridge().translate("unexpected_error") +
						//fileNameWithPath + ": " + 
						ReadResources.getLastErrorMessage() + "\n");
				continue;
			}

			CipherStuff.setCipherAlgo(oldCipher);
			HashStuff.setHashAlgo(oldHash);
			TestLog.v(ContentUtilities.class, "Temporarely set cipher to: " + CipherStuff.getCipherAlgo().getAlgorithmName()
					+ " and Hash to: " + HashStuff.getHashAlgo().getAlgorithmName());
			byte[] plainText = CipherStuff.getInstance().decrypt( oldEncrypted, 
					Arrays.copyOf(previousKeyAndAlgos[0], previousKeyAndAlgos[0].length), false);
			if (plainText == null) {
				TestLog.e(ContentUtilities.class, "Decryption failed: " + fileNameWithPath);
				TestLog.ve(ContentUtilities.class, CipherStuff.getErrorMessage(), 5);
				errorBuilder.append(PeaProperties.getVmBridge().translate("decryption_failed") + ": "+ fileNameWithPath + "\n");
				continue;
			}
			CipherStuff.setCipherAlgo(newCipher);
			HashStuff.setHashAlgo(newHash);
			TestLog.v(ContentUtilities.class, "Reset cipher to: " + CipherStuff.getCipherAlgo().getAlgorithmName()
					+ " and Hash to: " + HashStuff.getHashAlgo().getAlgorithmName());
			byte[] newEncrypted = CipherStuff.getInstance().encrypt(plainText, 
					Arrays.copyOf(newKeyAndAlgos[0], newKeyAndAlgos[0].length), false, null);
			if (newEncrypted == null) {
				TestLog.e(ContentUtilities.class, "Encryption failed: " + fileNameWithPath);
				TestLog.ve(ContentUtilities.class, CipherStuff.getErrorMessage(), 5);
				errorBuilder.append(PeaProperties.getVmBridge().translate("encryption_failed") + ": "+ fileNameWithPath + "\n");
				continue;
			}
			boolean writeSuccess = WriteResources.write(newEncrypted, fileName, dirName);
			if (writeSuccess == false) {
				TestLog.e(ContentUtilities.class, "Write failed: " + fileNameWithPath);
				TestLog.ve(ContentUtilities.class, WriteResources.getLastErrorMessage(), 5);
				errorBuilder.append(PeaProperties.getVmBridge().translate("unexpected_error") 
						+ fileNameWithPath + WriteResources.getLastErrorMessage() + "\n");
				continue;
			} else {
				TestLog.o(ContentUtilities.class, "Re-encrypted: " + fileNameWithPath);
				if (fileNameWithPath.startsWith(CloudControl.getTmpDir())) {
					// remove old file from properties
					int removedIndex = CloudPropertyHandler.removeFileFromAnyProviderSet(
							CloudControl.getProviderNameFromTmpFile(fileNameWithPath), 
							CloudControl.getFileNameFromTmpFile(fileNameWithPath),  null);
					if (removedIndex >= 0) {
						TestLog.v(ContentUtilities.class, fileNameWithPath + ": file removed from sub-set index " + removedIndex);
					} else if (removedIndex == -1) {
						TestLog.v(ContentUtilities.class, "File to removed from properties was not found: " + fileNameWithPath);
					} else if (removedIndex == -2) {
						TestLog.ve(ContentUtilities.class, "Error occurred when remiving filw from properties " + fileNameWithPath);
					}
					TestLog.v(ContentUtilities.class, "Upload re-encrypted file " + fileNameWithPath + "... ");
					saveTmpFileInCloud(fileNameWithPath);
				} else {
					//TestLog.v(ContentUtilities.class, "local file");
				}
			}			
		}
		CipherStuff.setCipherAlgo(currentCipher); // reset
		if (errorBuilder.length() > 0) {
			return new String(errorBuilder);
		} else {
			return null;
		}
	}

	/**
	 * Get the error message of the last decryption process
	 * 
	 * @return the error message as String or null
	 */
	public static String getDecryptErrorMessage() {
		return decryptErrorMessage;
	}
}
