package cologne.eck.all_peas.control;

/*
 * Peafactory - Production of Password Encryption Archives
 * Copyright (C) 20221  Axel von dem Bruch
 * 
 * This library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published 
 * by the Free Software Foundation; either version 2 of the License, 
 * or (at your option) any later version.
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE. 
 * See the GNU General Public License for more details.
 * See:  http://www.gnu.org/licenses/gpl-2.0.html
 * You should have received a copy of the GNU General Public License 
 * along with this library.
 */

import java.awt.Component;
import java.io.File;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.HashSet;
import java.util.ArrayList;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;

import cologne.eck.all_peas.data.PeaProperties;
import cologne.eck.all_peas.gui.PeaDialog;
import cologne.eck.all_peas.gui.PeaLockFrame;
import cologne.eck.all_peas.gui.PswDialogView;
import cologne.eck.all_peas.vm_specific.JREProperties;
import cologne.eck.tools.Converter;
import cologne.eck.tools.ExceptionHelper;
import cologne.eck.tools.PropertyHandler;
import cologne.eck.tools.ReadResources;
import cologne.eck.tools.StringNumberHelper;
import cologne.eck.tools.TestLog;
import cologne.eck.tools.UnexpectedValueException;
import cologne.eck.tools.WebTools;

public class UpdateChecker {
	
	private static File testFile = null;
	protected static void setTestFile(File _testFile) {	testFile = _testFile;}	
	
	private static String localFolder = null;//System.getProperty("user.dir") + File.separator + "resources" + File.separator + lang + File.separator;			
	private static String urlFolder = "https://eck.cologne/peafactory/src/language/";
// for tests: 
//	private static String urlFolder = "https://eck.cologne/peafactory/src/language-test/";// TODO test folder

	//private static boolean peaSpecificOnly = false;
	
	//public static boolean checkSuitableLanguageVersion() {
		
		// all_peas and specific: if same or higher version
		// and calendar is not version 1.8
		// an contact is not version 0.5
	//}

	/**
	 * Checks if there are updates of the currently installed language bundles. 
	 * Asks the user to install if available. Downloads then bundles in external folder "resources"
	 * 
	 * @return	true if any update is available
	 */
	public static boolean checkLanguageUpdates() {
		//public static boolean checkLanguageUpdates() {
		String[][] availableLangs = checkInstalledLanguages();
		boolean peaSpecificOnly = false;
		if (availableLangs != null && availableLangs[1] == null) { // do not install all PEA bundles
			peaSpecificOnly = true;
		}
		if (availableLangs == null || 
				( (availableLangs[0] == null || availableLangs[0].length == 0)
				&& 
				(availableLangs[1] == null || availableLangs[1].length == 0)) ) {
			TestLog.v(UpdateChecker.class, "No language update found");
			return false;
		}

		String availableLangString = "";
		if (availableLangs[0] != null) {
			for (String l : availableLangs[0]) {
				if (l.equals("") &&  ! availableLangString.contains(" en")) {
					l = "en";
				} 
				availableLangString = availableLangString + " " + l;
			}
		}
		if (availableLangs[1] != null) {
			for (String l : availableLangs[1]) {
				if (l.equals("") &&  ! availableLangString.contains(" en")) {
					l = "en";
				}
				if ( ! availableLangString.contains(l)) {
					availableLangString = availableLangString + " " + l;
				}
			}
		}
		int installLang = PeaDialog.showQuestion(JREProperties.getMainWindow(),  
				PeaProperties.getVmBridge().translate("update_found") + ":\n" 
						+ PeaProperties.getVmBridge().translate("language") + availableLangString + "\n\n"
						+ PeaProperties.getVmBridge().translate("install") + "?", 
						PeaProperties.getVmBridge().translate("update_found"), 0);
		if (installLang == 0) {
			HashSet<String> langsToInstall = new HashSet<String>();
			if (availableLangs[0] != null) {
				langsToInstall.addAll(Converter.arrayToHashSet( availableLangs[0]));
			}
			if (availableLangs[1] != null) {
				langsToInstall.addAll(Converter.arrayToHashSet( availableLangs[1]));
			}
			//HashSet<String> installedLangs = PeaProperties.getVmBridge().getTranslator().getExternalInstalledLanguages();
			for (String lang : langsToInstall) {
				String error = downloadLanguageFiles(lang, true, peaSpecificOnly);					
				if (error != null) {
					PeaControl.setMessage( error, true);
				}
			}
			// Update the view: 
			if (JREProperties.getMainWindow() instanceof PswDialogView) {
				PswDialogView.resetLanguage();
			} else if (JREProperties.getMainWindow() instanceof PeaLockFrame) {
				((PeaLockFrame) JREProperties.getMainWindow()).resetFrame();
			}
		}
		return true;
	}

	/**
	 * Check if there is an update available of the current PEA 
	 * (read from PAD file). 
	 * 
	 * @param c				the Component
	 * @param showErrors	true: show errors as dialogs
	 * @param checkLanguages	true: check for language bundle updates
	 */
	public static void checkUpdates(Component c, boolean showErrors, boolean checkLanguages){
		
		TestLog.v(UpdateChecker.class, "Check updates... " );

		if (checkLanguages == true) {
			checkLanguageUpdates();
		}
		// create PAD file url: 
		// e.g. https://eck.cologne/peafactory/src/PAD/calendarPAD.xml
		String baseUrl = "https://eck.cologne/peafactory/src/PAD/";
		String peaExtension = null;
		String appType = PeaProperties.getFileType();
		if (appType.equals("calendar")) {
			peaExtension = "calendarPAD.xml";
		} else if (appType.equals("note")) {
			peaExtension = "notebookPAD.xml";
		} else if (appType.equals("file")) {
			peaExtension = "filePAD.xml";
		} else if (appType.equals("image")) {
			peaExtension = "imagePAD.xml";
		} else if (appType.equals("contact")) {
			peaExtension = "contactPAD.xml";
		} else {
			new UnexpectedValueException("String", "appType", "is not valid: " + appType).printDescription();
			if (showErrors == true) {
				PeaDialog.showMessage(c, PeaProperties.getVmBridge().translate("unexpected_error"), 
						PeaProperties.getVmBridge().translate("error"), 0);
			}
			return;
		}
		String urlString = baseUrl + peaExtension;
		if (WebTools.checkConnection() == false){
			if (showErrors == true) {
				PeaDialog.showMessage(c, PeaProperties.getVmBridge().translate("bad_web_connection"), 
						PeaProperties.getVmBridge().translate("error"), 0);
			}
			return;
		}
		TestLog.v(UpdateChecker.class, "Check URL: " + urlString);
		File file = WebTools.getFileFromURL(urlString);
		// for testing:
		if (testFile != null) {
			file = testFile;
		}
		
		if (file == null) {
			if (showErrors == true) {
				PeaDialog.showMessage(c, PeaProperties.getVmBridge().translate("download_failed"), 
						PeaProperties.getVmBridge().translate("error"), 0);
			}
			return;				
		} else {
			DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory
					.newInstance();
			DocumentBuilder documentBuilder;
			try {
				documentBuilder = documentBuilderFactory.newDocumentBuilder();
				Document document = documentBuilder.parse(file);
				
				//TestLog.v(UpdateChecker.class, "Available version... " );
				String availableVersion = "";
				if (document != null
						&& document.getElementsByTagName("Program_Version") != null 
						&& document.getElementsByTagName("Program_Version").item(0) != null 
						&& document.getElementsByTagName("Program_Version").item(0).getTextContent() != null) {
					availableVersion = document.getElementsByTagName("Program_Version").item(0).getTextContent();
					if (availableVersion == null) {
						TestLog.e(UpdateChecker.class, "Missing available version...");
						if (showErrors == true) {
							PeaDialog.showMessage(c, PeaProperties.getVmBridge().translate("unexpected_error"), 
									PeaProperties.getVmBridge().translate("error"), 0);
						}
						return;
					}
				} else {
					TestLog.e(UpdateChecker.class, "Missing available version...");
					if (showErrors == true) {
						PeaDialog.showMessage(c, PeaProperties.getVmBridge().translate("unexpected_error"), 
								PeaProperties.getVmBridge().translate("error"), 0);						
					}
					return;
				}
				//TestLog.v(UpdateChecker.class, "Available version: " + availableVersion);
				// get release status: Major Update|Minor Update|New Release|Beta|Alpha|Media Only
				String releaseStatus = document.getElementsByTagName("Program_Release_Status").item(0).getTextContent();
				if (releaseStatus == null) {
					releaseStatus = "";
				}
				//TestLog.v(UpdateChecker.class, "Release: " + releaseStatus);
				String currentVersion = PeaControl.getDialog().getVersion();
				//String currentVersionNumber = currentVersion.replaceAll("[^0-9.]", "");
				TestLog.o(UpdateChecker.class, "current version: " + currentVersion
						+ ", available version: " + availableVersion + ", release status: " + releaseStatus);

				int newVersion = StringNumberHelper.checkNewVersionNumber(currentVersion, availableVersion);
				//TestLog.o(UpdateChecker.class, "newVersion: " + newVersion);
				if (newVersion == 1) {
					// always show new version
				} else if (newVersion == 0) {
					// in current version string contains possibly alpha or beta
					// check release status:
					if (currentVersion.toLowerCase().contains("alpha")) {
						if ( releaseStatus.toLowerCase().equals("alpha")) {
							if (showErrors == true) {
								PeaDialog.showMessage(c, PeaProperties.getVmBridge().translate("no_update_found"));
							}
							return;
						} else {
							// show new version
						}
					} else if (currentVersion.contains("beta")) {
						if ( releaseStatus.toLowerCase().equals("beta") || releaseStatus.toLowerCase().equals("alpha")) {
							if (showErrors == true) {
								PeaDialog.showMessage(c, PeaProperties.getVmBridge().translate("no_update_found"));
							}
							return;
						} else {
							// show new version
						}		
					} else {
						if (showErrors == true) {
							PeaDialog.showMessage(c, PeaProperties.getVmBridge().translate("no_update_found"));
						}
						return;
					}
				} else {
					if (showErrors == true) {
						PeaDialog.showMessage(c, PeaProperties.getVmBridge().translate("no_update_found"));
					}
					return;
				}
				// TODO date of new version, changes...
				String[] messageAndLink = {PeaProperties.getVmBridge().translate("web"), 
						//  Invalid link until peafactory 0.4.7 with trailing /
						// "http://eck.cologne/peafactory/en/" + PeaControl.getDialog().getWebsite()
						"http://eck.cologne/peafactory/en" + PeaControl.getDialog().getWebsite()
				};
				String[][] messagesAndLinks = {messageAndLink};
				PeaDialog.showMessageWithLink(c, 
						PeaProperties.getVmBridge().translate("update_found") + "\n      " 
								+ PeaProperties.getVmBridge().translate("version") + " " + availableVersion//av
									+ " (" + releaseStatus + ")",
								messagesAndLinks, null, -1);
			} catch (ParserConfigurationException e) {
				TestLog.e(UpdateChecker.class, new ExceptionHelper(e).getInfos());
				if (showErrors == true) {
					PeaDialog.showMessage(c, PeaProperties.getVmBridge().translate("unexpected_error"), 
							PeaProperties.getVmBridge().translate("error"), 0);
				}
			} catch (Exception e) {
				TestLog.e(UpdateChecker.class, e.toString() + " - " + e.getLocalizedMessage());

				if (showErrors == true) {
					PeaDialog.showMessage(c, PeaProperties.getVmBridge().translate("unexpected_error"), 
							PeaProperties.getVmBridge().translate("error"), 0);
				}
			}
		}		
	}
	/**
	 * Compare two version numbers. All non numeric characters are ignored.
	 * 
	 * @param version1
	 * @param version2
	 * 
	 * @return 0 for equal,
	 * 			-1 is version1 is less than version2, 
	 * 			1 if version1 is greater than version2
	 */
/*	private static int compareVersions(String version1, String version2) {
		// replace all non numeric characters
		version1 = version1.replaceAll("[^\\d.]", "");
		version2 = version2.replaceAll("[^\\d.]", "");
		String[] v1 = version1.split(".");
		String[] v2 = version2.split(".");
		int index = 0;
		while (index > v1.length && index > v2.length) {
			if (Integer.parseInt(v1[index]) > Integer.parseInt(v2[index])) {
				return 1;
			} else if (Integer.parseInt(v1[index]) < Integer.parseInt(v2[index])) {
				return -1;
			}
		}
		// Check if there is a sub-version: 
		if (v1.length > v2.length) {
			return 1;
		} else if (v1.length < v2.length) {
			return -1;
		}
		return 0;
	}*/
	
	/**
	 * Check if there is an update of language packs. 
	 * Returns an array of language short cuts of available updates.
	 * 
	 * @return	null for no update, otherwise an array in form of 
	 * 			array[0] array of pea-specific bundles, array[1] array of general bundles
	 */
	private static String[][] checkInstalledLanguages() {
		
		//peaSpecificOnly = false;
		
		if (WebTools.checkConnection() == false) {
			String errorMessage = PeaProperties.getVmBridge().translate("bad_web_connection");
			TestLog.ve(UpdateChecker.class, "No web connection available for language update... ");
			PeaControl.setMessage( errorMessage, true);
			return null;
		}

		String currentVersion = PeaControl.getDialog().getVersion(); // e.g. 1.8 (PEA version 2)
		currentVersion = currentVersion.substring(0, currentVersion.indexOf("(")).trim(); // cut PEA version note
		
		String type = PeaProperties.getFileType();
		
		String newPeaSpecific = null; // new version of this specific PEA
		try {
			newPeaSpecific = WebTools.readPropertyFromWeb(urlFolder + "lang.properties", type + "_version_specific").trim();
		} catch (Exception e) {
			TestLog.ve(UpdateChecker.class, "Language update check failed. Can't read properties from web: " + e.getLocalizedMessage());
		}
		if (newPeaSpecific == null) {
			String errorMessage = PeaProperties.getVmBridge().translate("operation_failed") + " - " 
					+ PeaProperties.getVmBridge().translate("language");
			TestLog.ve(UpdateChecker.class, "Language update check failed. Missing property from web... ");
			PeaControl.setMessage( errorMessage, true);
			return null;
		}
		// do not install bundles from further or older versions
		if (currentVersion.equals(newPeaSpecific) == false) {
			TestLog.v(UpdateChecker.class, "Version of PEA differs: " + currentVersion + " - " + newPeaSpecific);
			return null;
		}
		//TestLog.v(UpdateChecker.class, "Version of PEA: " + currentVersion + " - " + newPeaSpecific);

		// for return value: 
		ArrayList<String> specificBundles = new ArrayList<String>(); // PEA specific
		ArrayList<String> generalBundles = new ArrayList<String>(); // for all PEAs
		String[][] returnValue = new String[2][];
	
		// local folder: 
		if (localFolder == null) {		
			localFolder = System.getProperty("user.dir") + File.separator + "resources" + File.separator;
			// is static: urlFolder = "https://eck.cologne/peafactory/src/language/";
		}		
		
		// Check specific language bundles (specific for this PEA): Never null, but maybe an empty set
		HashSet<String> installedLangs = PeaProperties.getVmBridge().getTranslator().getAvailableLanguagesAsSet(true);
		// Always check default language English:
		if (checkBundleUpdate(type, "", true) == true) {
			specificBundles.add("");
		}
		// Check all installed languages for PEA-specific language bundle updates:
		for (String installedLang : installedLangs) {
			if (checkBundleUpdate(type, installedLang, true) == true) { 
				specificBundles.add(installedLang);
			}
		}
				
		// Check general language bundles (bundles for all PEAs):
		String newPeaGeneral = null; // required PeaFactoty version of this new PEA
		String newGeneral = null; // new PeaFactoy version
		try {
			newPeaGeneral = WebTools.readPropertyFromWeb(urlFolder + "lang.properties", type + "_version_general").trim();
			newGeneral = WebTools.readPropertyFromWeb(urlFolder +"lang.properties", "version_general").trim();
		} catch (Exception e) {
			String errorMessage = PeaProperties.getVmBridge().translate("operation_failed") + " - " 
					+ PeaProperties.getVmBridge().translate("language");
			TestLog.ve(UpdateChecker.class, "Language update check failed. Missing property from web... ");
			PeaControl.setMessage( errorMessage, true);
			if (specificBundles != null && specificBundles.size() > 0) {
				returnValue[0] = Converter.arrayListToArray(specificBundles);
			}
			return returnValue;
		}
		// Do not install general bundles from other PeaFactory versions:
		if ( ! newPeaGeneral.equals(newGeneral)) {
			TestLog.v(UpdateChecker.class, "No update for general bundle: other PEA version: "
					+ newPeaGeneral + " - " + newGeneral);
			if (specificBundles != null && specificBundles.isEmpty() == false) {
				returnValue[0] = Converter.arrayListToArray(specificBundles);
				return returnValue;
			} else {
				return null;
			}
		}
		
		// Always check default language English:
		if (checkBundleUpdate(type, "", false) == true) {
			generalBundles.add("");
		}
		// Check all installed languages for non PEA-specific language bundle updates:
		for (String installedLang : installedLangs) {
			if (checkBundleUpdate(type, installedLang, false) == true) { 
				generalBundles.add(installedLang);
			}
		}
		if (specificBundles.isEmpty() && generalBundles.isEmpty()) {
			return null; // no update available
		} else {
			returnValue[0] = Converter.arrayListToArray(specificBundles);
			returnValue[1] = Converter.arrayListToArray(generalBundles);
		}
		return returnValue;		
	}
	
	/**
	 * Construct a path for language bundles in file system
	 * and in web
	 * 
	 * @param type			type of PEA (calendar,contact,image,note)
	 * @param lang			language short cut
	 * @param peaSpecific	true if PEA-specific bundle
	 * 
	 * @return array of paths: [0] is localPath, [1] is urlPath
	 */
	private static String[] constructLocalAndWebPath(String type, String lang, boolean peaSpecific) {
		
		// Construct the local and web path:
		String localPath = null; // path in file system
		String urlPath = null;  // path in web
		String langExtension = "";// extension before .properties in file name: e.g. _de
		if (lang.equals("en")) { // default bundle
			lang = "";
		}
		if (lang.equals("")) {
			localPath = localFolder;			
			urlPath = urlFolder;
		} else {
			localPath = localFolder + lang + File.separator;			
			urlPath = urlFolder + lang +"/";
			langExtension = "_" + lang;
		}
		if (peaSpecific == true) {
			// first letter of type must be upper case: e.g. calendar -> Calendar
			String cap = type.substring(0, 1).toUpperCase() + type.substring(1);
			localPath = localPath + cap + "LanguageBundle" + langExtension + ".properties";
			urlPath = urlPath + cap + "LanguageBundle" + langExtension + ".properties";
		} else {
			localPath =  localPath + "PeaLanguagesBundle" + langExtension + ".properties";
			urlPath =  urlPath + "PeaLanguagesBundle" + langExtension + ".properties";
		}	
		String[] result = new String[2];
		result[0] = localPath;
		result[1] = urlPath;
		return result;
	}
	
	/**
	 * Checks if an update of the bundle is available in web. 
	 * 
	 * @param type			the type of the PEA: contact, calendar, image, note (!)
	 * @param lang			the language short cut e.g. de, fr, tr
	 * @param peaSpecific	true for PEA-specific bundle
	 * 
	 * @return	true if update was found
	 */
	private static boolean checkBundleUpdate(String type, String lang, boolean peaSpecific) {
		//TestLog.v(UpdateChecker.class, "checkBundleUpdate: " + lang + " - " + type + " peaSpecific: " + peaSpecific);
		if (type == null || lang == null) {
			TestLog.e(UpdateChecker.class, "Unexpected error", 5);
			return false;
		}	
//		if (type.equals("file") && peaSpecific == true) { // There is no specific file bundle // TODO change if there is one
//			return false;
//		}
		String[] paths = constructLocalAndWebPath(type, lang, peaSpecific);
		String localPath = paths[0];
		String urlPath = paths[1];

		// Create web URL:
		URL url = null;
		try {
			URI uri = new URI(urlPath);
			url = uri.toURL(); //  get URL from URI
		} catch (MalformedURLException e) {
			TestLog.ve(UpdateChecker.class, "Invalid URL: " + urlPath);
			return false;
		} catch (URISyntaxException e) {
			TestLog.ve(UpdateChecker.class, "Invalid URL (URI): " + urlPath);
			return false;
		}
		// Check file existence, get file size to compare: 
		File file = new File(localPath);
		long fileSize = -1;
		if (file.exists()) { // check first in external folder resources
			fileSize = file.length();
		} else {
			// check inside jar: cut first part
			byte[] contentInJar = ReadResources.getResourceFromJAR(
					"resources/" + localPath.substring(localFolder.length(), localPath.length()));
			//TestLog.v(UpdateChecker.class, "Check inside jar: " + "resources/" + localPath.substring(localFolder.length(), localPath.length()));
			if (contentInJar == null) {
				TestLog.ve(UpdateChecker.class, "File access in jar failed: "
						+ "resources/" + localPath.substring(localFolder.length(), localPath.length()), 5);
				return false;
			} else {
				fileSize = contentInJar.length;
			}
		}
		HttpURLConnection conn = null;
		try {
			conn = (HttpURLConnection) url.openConnection();
			conn.setRequestMethod("HEAD");
			long webSize = conn.getContentLengthLong();
			if (webSize == -1) {
				TestLog.ve(UpdateChecker.class, "Get size from web file failed: " + url.toString());
				return false;
			}
			if (webSize != fileSize) {
				///// Update found /////
				TestLog.v(UpdateChecker.class, url.toString() 
						+ " - Language update found: locale size: " + fileSize 
						+ ", web size: " + conn.getContentLengthLong());
				//TestLog.e(UpdateChecker.class, "Update available: " + lang + ", webSize: " + webSize + ", fileSize: " + fileSize);
				return true;
			} else {
				//TestLog.e(UpdateChecker.class, "No update available: " + lang + ", webSize: " + webSize + ", fileSize: " + fileSize);
			}
		} catch (IOException e) {
			TestLog.ve(UpdateChecker.class, url.toString() + " - Failed to compare installed file with file in web. " + e.getLocalizedMessage());
			return false;
		} finally {
			if (conn != null) {
				conn.disconnect();
			}
		}
		return false;
	}
	
	/**
	 * Download language files and set language if 
	 * download was successful. The view should be updated, if the
	 * download was successful. 
	 * 
	 * @param language	the language to download
	 * @param updateLanguage	true, if an update of an already installed language
	 * 							is to be downloaded, false for new language
	 * @param peaSpecificOnly	true: only PEA-specific bundle, not PeaLaguage.properties
	 * 
	 * @return	null for success, an error message otherwise
	 */
	public static String downloadLanguageFiles(String language, boolean updateLanguage, boolean peaSpecificOnly) {

		String error = null;
		// local folder: 
		if (localFolder == null) {		
			localFolder = System.getProperty("user.dir") + File.separator + "resources" + File.separator;
			// is static: urlFolder = "https://eck.cologne/peafactory/src/language/";
		}
		String type = PeaProperties.getFileType();
		String localPath = null;
		String urlPath = null;

		if (peaSpecificOnly == false) {
			String[] nonPeaSpecificPaths = constructLocalAndWebPath(type, language, false);
			localPath = nonPeaSpecificPaths[0];
			urlPath = nonPeaSpecificPaths[1];
			// Download file in folder resources
			try {
				// before peafactory 0.1.4.4:
				//String urlFolder = "https://eck.cologne/peafactory/src/lang/" + langs[langIndex];
				// from peafactory.0.1.4.4
				//	String urlFolder = "https://eck.cologne/peafactory/src/language/" + language;
				long byteNumber = WebTools.downloadFile(urlPath, localPath);//  urlFolder + "/" + urlFile, localFolder + File.separator + urlFile);
				//System.getProperty("user.dir") + File.separator + "resources" + File.separator + "PeaLanguagesBundle_" + langs[langIndex] + ".properties");
				if (byteNumber == 0) {
					TestLog.e(UpdateChecker.class, "Download failed");
					TestLog.ve(UpdateChecker.class, "Download failed", 5);
					PeaDialog.showMessage(JREProperties.getMainWindow(), 
							PeaProperties.getVmBridge().translate("download_failed")
							+ "\n" + urlPath, 
							PeaProperties.getVmBridge().translate("error"), 0);
					error = PeaProperties.getVmBridge().translate("download_failed") + "  " + language;
				}
			} catch (Exception e) {
				TestLog.e(UpdateChecker.class, e.getLocalizedMessage());
				TestLog.ve(UpdateChecker.class, e.getLocalizedMessage() + " - " + e.toString(), 5);
				PeaDialog.showMessage(JREProperties.getMainWindow(), 
						language + ": " + PeaProperties.getVmBridge().translate("download_failed") + "\n" 
								+ e.getLocalizedMessage(), 
								PeaProperties.getVmBridge().translate("error"), 0);
				error = e.getLocalizedMessage();
			}
		}

		// download PEA-specific bundles:			
		String[] peaSpecificPaths = constructLocalAndWebPath(type, language, true);
		localPath = peaSpecificPaths[0];
		urlPath = peaSpecificPaths[1];
		// Download file in folder resources
		try {
			// before peafactory 0.1.4.4:
			//String urlFolder = "https://eck.cologne/peafactory/src/lang/" + langs[langIndex];
			// from peafactory.0.1.4.4
			//	String urlFolder = "https://eck.cologne/peafactory/src/language/" + language;
			long byteNumber = WebTools.downloadFile(urlPath, localPath);//  urlFolder + "/" + urlFile, localFolder + File.separator + urlFile);
			//System.getProperty("user.dir") + File.separator + "resources" + File.separator + "PeaLanguagesBundle_" + langs[langIndex] + ".properties");
			if (byteNumber == 0) {
				TestLog.e(UpdateChecker.class, "Download failed");
				TestLog.ve(UpdateChecker.class, "Download failed", 5);
				PeaDialog.showMessage(JREProperties.getMainWindow(), 
						PeaProperties.getVmBridge().translate("download_failed")
						+ "\n" + urlPath, 
						PeaProperties.getVmBridge().translate("error"), 0);
				error = PeaProperties.getVmBridge().translate("download_failed") + "  " + language;
			}
		} catch (Exception e) {
			TestLog.e(UpdateChecker.class, e.getLocalizedMessage());
			TestLog.ve(UpdateChecker.class, e.getLocalizedMessage() + " - " + e.toString(), 5);
			PeaDialog.showMessage(JREProperties.getMainWindow(), 
					language + ": " + PeaProperties.getVmBridge().translate("download_failed") + "\n" 
							+ e.getLocalizedMessage(), 
							PeaProperties.getVmBridge().translate("error"), 0);
			error = e.getLocalizedMessage();
		}
		// update setting menu 
		if (updateLanguage == false) { 
			// new language: Change to this language
			try {
				PeaProperties.getVmBridge().setLanguagesBundle(language);
			} catch (Exception e) {
				TestLog.e(UpdateChecker.class, e.getLocalizedMessage());
				TestLog.ve(UpdateChecker.class, e.getLocalizedMessage() + " - " + e.toString(), 5);
				PeaDialog.showMessage(JREProperties.getMainWindow(), e.getLocalizedMessage() + "\n"
						+ PeaProperties.getVmBridge().translate("language") + ": " + language, 
						PeaProperties.getVmBridge().translate("error"), 0);
			}
			PeaProperties.setLastLanguage(language);
			new PropertyHandler().setOneProperty("language", language);
		} else {
			// TODO Installing several languages results in several updates and setting of last language
		}
		return error;
	}
	/**
	 * Get the web folder for language updates: 
	 * "https://eck.cologne/peafactory/src/language/";
	 *  or for tests:  "https://eck.cologne/peafactory/src/language-test/"
	 * 
	 * @return the web folder for languages
	 */
	public static String getLanguageUrlFolder() {
		return urlFolder;
	}
}
