package cologne.eck.all_peas.gui;

/*
 * PeaFactory - Production of Password Encryption Archives
 * Copyright (C) 2015  Axel von dem Bruch
 * 
 * This library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published 
 * by the Free Software Foundation; either version 2 of the License, 
 * or (at your option) any later version.
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE. 
 * See the GNU General Public License for more details.
 * See:  http://www.gnu.org/licenses/gpl-2.0.html
 * You should have received a copy of the GNU General Public License 
 * along with this library.
 */
import java.awt.Image;
import java.awt.Toolkit;
import java.io.File;
import java.util.ArrayList;

import javax.swing.ImageIcon;

import cologne.eck.all_peas.data.PeaProperties;
import cologne.eck.tools.TestLog;
import settings.PeaSettings;

public class IconManager {
	
	private static ArrayList<Image> imageList = null;
	
	/**
	 * Load an ImageIcon for javax swing components
	 * from an image file. The image file can be inside 
	 * the jar archive or in the file system, but must
	 * be in the directory resources.
	 * 
	 * @param imageFileName		the file name of the image file
	 * 							in the directory resources
	 * 							( png, gif,jpg, bmp)
	 * @param description		the description of the icon
	 * @param accessibleName	name of the icon, e.g. for icon-only 
	 * 							buttons
	 * @param accessibleDescription		the description of the icon, 
	 * 									describing the purpose of the object
	 * 
	 * @return	the ImageIcon loaded from the file
	 */
	public static ImageIcon loadIcon(String imageFileName, String description,
			String accessibleName, String accessibleDescription) {
		
		ImageIcon icon = null;
		// first try to load from jar archive
		try {
			icon = new ImageIcon(
					IconManager.class.getClassLoader().getResource("resources/"+ imageFileName));//, 
			//		description);
		} catch (Exception e) {
			// try to load from file system: 
			icon = new ImageIcon("resources" + File.separator + imageFileName);
			// check success:
			if (icon.getImageLoadStatus() == 4){// MediaTracker.ERRORED
				icon = new ImageIcon(Toolkit.getDefaultToolkit().createImage("resources/" + imageFileName));
				// check success:
				if (icon.getImageLoadStatus() == 4){// MediaTracker.ERRORED
					// try to load from jar
					java.net.URL imgURL = IconManager.class.getResource("/resources/"+ imageFileName);
					if (imgURL != null) {
						icon = new ImageIcon(imgURL);
					} else {
						icon = null;
					}
				}
			}
		}
		if (icon == null) {
			TestLog.ve(IconManager.class,"is null: " + imageFileName, 5);
			return null;
		} else if (icon.getImageLoadStatus() == 4){// MediaTracker.ERRORED
			TestLog.ve(IconManager.class,"ImageIcon s not loaded: " + imageFileName, 5);
			return null;
		} else {
			icon.setDescription(description);
		}
		if (accessibleName != null) {
			icon.getAccessibleContext().setAccessibleDescription(accessibleName);
		}
		if (accessibleDescription != null) {
			icon.getAccessibleContext().setAccessibleDescription(accessibleDescription);
		}
		return icon;
	}
	
	/**
	 * Load a scaled ImageIcon for javax swing components
	 * from an image file. The image file can be inside 
	 * the jar archive or in the file system, but must
	 * be in the directory resources.
	 * 
	 * @param imageFileName		the file name of the image file
	 * 							in the directory resources
	 * 							( png, gif,jpg, bmp)
	 * @param description		the description of the icon
	 * @param width				the intended width if the icon
	 * @param height			the intended height of the icon
	 * @param accessibleName	name of the icon, e.g. for icon-oly 
	 * 							buttons
	 * @param accessibleDescription		the description of the icon, 
	 * 									describing the purpose of the object
	 * 
	 * @return	the ImageIcon loaded from the file
	 */
	public static ImageIcon loadScaledIcon(String imageFileName, String description, 
			int width, int height,
			String accessibleName, String accessibleDescription) {
		
		ImageIcon icon = null;
		// first try to load from jar archive
		try {
			icon = new ImageIcon(
					IconManager.class.getClassLoader().getResource("resources/"+ imageFileName));//, 
			//		description);
		} catch (Exception e) {
			// try to load from file system: 
			icon = new ImageIcon("resources" + File.separator + imageFileName);
			// check success:
			if (icon.getImageLoadStatus() == 4){// MediaTracker.ERRORED
				icon = new ImageIcon(Toolkit.getDefaultToolkit().createImage("resources/" + imageFileName));
				// check success:
				if (icon.getImageLoadStatus() == 4){// MediaTracker.ERRORED
					// try to load from jar
					java.net.URL imgURL = IconManager.class.getResource("/resources/"+ imageFileName);
					if (imgURL != null) {
						icon = new ImageIcon(imgURL);
					} else {
						icon = null;
					}
				}
			}
		}
		if (icon == null) {
			TestLog.ve(IconManager.class,"is null: " + imageFileName, 5);
			return null;
		} else if (icon.getImageLoadStatus() == 4){// MediaTracker.ERRORED
			TestLog.ve(IconManager.class, "ImageIcon is not loaded: " + imageFileName, 5);
			return null;
		} else {
			icon.setDescription(description);
		}
		Image image = icon.getImage().getScaledInstance(width,height,Image.SCALE_DEFAULT);
		icon = new ImageIcon(image);
		if (accessibleName != null) {
			icon.getAccessibleContext().setAccessibleDescription(accessibleName);
		}
		if (accessibleDescription != null) {
			icon.getAccessibleContext().setAccessibleDescription(accessibleDescription);
		}
		return icon;
	}
	/**
	 * Load pea specific images and add to the icon list
	 * 
	 * @param fileName		the file name without path, with extension
	 * @param description	the description of the image
	 */
	private static void addPeaSpecificImagesToList(String fileName, String description) {

		if (fileName == null) {
			TestLog.ve(IconManager.class, "PEA specific image is null...", 5);
			return;
		}
		// load and add image if found
		ImageIcon peaSpecificIcon = IconManager.loadIcon(fileName, 
				null, 
				null, 
				description);
		if (peaSpecificIcon != null) {
			Image peaSpecific = peaSpecificIcon.getImage();
			imageList.add(peaSpecific);
		} else {
			TestLog.ve(IconManager.class, "Can't find PEA specific image: " + fileName, 5);
		}
	}
	
	/**
	 * Get the list of images to set as icons
	 * 
	 * @return	list of images
	 */
	public static synchronized ArrayList<Image> getIconList() {

		if ( imageList == null) {
			
			imageList = new ArrayList<Image>();

			Image pea16 = IconManager.loadIcon("pea16.png", 
					null, 
					null, 
					PeaProperties.getVmBridge().translate("logo_of_pea") ).getImage();
			Image pea20 = IconManager.loadIcon("pea20.png", 
					null, 
					null, 
					PeaProperties.getVmBridge().translate("logo_of_pea") ).getImage();
			Image pea24 = IconManager.loadIcon("pea24.png", 
					null, 
					null, 
					PeaProperties.getVmBridge().translate("logo_of_pea") ).getImage();
			if (pea16 != null) {
				imageList.add(pea16);
			}
			if (pea20 != null) {
				imageList.add(pea20);
			}
			if (pea24 != null) {
				imageList.add(pea24);
			}
			if (PeaSettings.getFILE_TYPE().equals("calendar")) { 
				
				addPeaSpecificImagesToList("calendar-pea-32.png", "calendar with pea logo");				
				addPeaSpecificImagesToList("calendar-pea-64.png", "calendar with pea logo");
				
	/*			Image peaSpecific32 = IconManager.loadIcon("calendar-pea-32.png", 
						null, 
						null, 
						PeaProperties.getVmBridge().translate("logo_of_pea") ).getImage();
				if (peaSpecific32 != null) {
					imageList.add(peaSpecific32);
				} else {
					TestLog.ve(IconManager.class, "Can't find PEA specific image: calendar-pea-64.png");
				}
				
				Image peaSpecific64 = IconManager.loadIcon("calendar-pea-64.png", 
						null, 
						null, 
						"calendar with pea logo" ).getImage();
				if (peaSpecific64 != null) {
					imageList.add(peaSpecific64);
				} else {
					TestLog.ve(IconManager.class, "Can't find PEA specific image: calendar-pea-64.png");
				} */
			} else if (PeaSettings.getFILE_TYPE().equals("contact")) { 
				addPeaSpecificImagesToList("contact-pea-32.png", "person with pea logo");				
				addPeaSpecificImagesToList("contact-pea-64.png", "person with pea logo");
				
	/*			ImageIcon peaSpecificIcon32 = IconManager.loadIcon("contact-pea-32.png", 
						null, 
						null, 
						"person with pea logo" );
				if (peaSpecificIcon32 != null) {
					Image peaSpecific32 = peaSpecificIcon32.getImage();
					imageList.add(peaSpecific32);
				} else {
					imageList.add(pea24);
					TestLog.ve(IconManager.class, "Can't find PEA specific image: contact-pea-32.png");
				}
				ImageIcon peaSpecificIcon = IconManager.loadIcon("contact-pea-64.png", 
						null, 
						null, 
						"person with pea logo" );
				if (peaSpecificIcon != null) {
					Image peaSpecific = peaSpecificIcon.getImage();
					imageList.add(peaSpecific);
				} else {
					imageList.add(pea24);
					TestLog.ve(IconManager.class, "Can't find PEA specific image: contact-pea-64.png");
				} */
			} else if (PeaSettings.getFILE_TYPE().equals("file")) { 
				addPeaSpecificImagesToList("file-pea-32.png", "file with pea logo");				
				addPeaSpecificImagesToList("file-pea-64.png", "file with pea logo");
			} else if (PeaSettings.getFILE_TYPE().equals("note")) { 
				addPeaSpecificImagesToList("notebook-pea-32.png", "notebook with pea logo");				
				addPeaSpecificImagesToList("notebook-pea-64.png", "notebook with pea logo");
			} else if (PeaSettings.getFILE_TYPE().equals("image")) { 
				addPeaSpecificImagesToList("image-pea-32.png", "image with pea logo");				
				addPeaSpecificImagesToList("image-pea-64.png", "image with pea logo");
			} else {
				TestLog.ve(IconManager.class, "Unknown type: " + PeaSettings.getFILE_TYPE());
			}
		}
		return imageList;
	}
}
