package cologne.eck.file_pea;


import java.awt.Dimension;
import java.awt.Point;

/*
 * Peafactory - Production of Password Encryption Archives
 * Copyright (C) 2015  Axel von dem Bruch
 * 
 * This library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published 
 * by the Free Software Foundation; either version 2 of the License, 
 * or (at your option) any later version.
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE. 
 * See the GNU General Public License for more details.
 * See:  http://www.gnu.org/licenses/gpl-2.0.html
 * You should have received a copy of the GNU General Public License 
 * along with this library.
 */

/**
 * Dialog to get the password and start the file pea.
 */

/*
 * Start File Lock PEA with password or keyfile but without files
 * on first start. Add files later
 * -> there is no password confirmation!!!
 */

import java.io.File;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Properties;

import javax.swing.SwingUtilities;
import cologne.eck.all_peas.control.CommandLineHandler;
import cologne.eck.all_peas.control.PeaControl;
import cologne.eck.all_peas.data.AttachmentHandler;
import cologne.eck.all_peas.data.Attachments;
import cologne.eck.all_peas.data.PeaProperties;
import cologne.eck.all_peas.files.FilePanelHolder;
import cologne.eck.all_peas.files.FileComposer;
import cologne.eck.all_peas.files.FileModel;
import cologne.eck.all_peas.files.FileTypePanel;
import cologne.eck.all_peas.gui.CursorManager;
import cologne.eck.all_peas.gui.KeyFileDialogs;
import cologne.eck.all_peas.gui.PeaDialog;
import cologne.eck.all_peas.gui.PeaLockFrame;
import cologne.eck.all_peas.gui.PswDialogView;
import cologne.eck.all_peas.util.TimeoutTimer;
import cologne.eck.all_peas.vm_specific.JREProperties;
import cologne.eck.peafactory.crypto.AlgoParamHandler;
import cologne.eck.peafactory.crypto.CipherStuff;
import cologne.eck.peafactory.crypto.RandomStuff;
import cologne.eck.peafactory.crypto.SessionKeyCrypt;
import cologne.eck.peafactory.crypto.kdf.KeyDerivation;
import cologne.eck.tools.ExceptionHelper;
import cologne.eck.tools.PropertyHandler;
import cologne.eck.tools.TestLog;
//import cologne.eck.tools.Help;
import cologne.eck.tools.UnexpectedValueException;


public final class FileControlStartWithoutFiles extends PeaControl {

	private static volatile FileControlStartWithoutFiles pswDialog; // static instance
	private LockFrameFile lockFrame; // associated static instance of LockFrameFile
	//private static PswDialogView dialogView; // associated instance of PswDialogView
	private final static String peaPropFileName = "file.properties";
		
	/**
	 * used for ShutDownHook (if program is not closing correctly)
	 * true = unencrypted files remaining (something to do for ShutDownHook) 
	 * false = all files encrypted (nothing to do for ShutDownHook)
	 */
	private static volatile boolean decrypted = false; 
	
	private static final String[] supportedLanguages = {"cz", "de", "es", "fr", 
			"gr", "id", "it", "lv", "nl", "pt", "ro",  "ru", "tr", 
			 "cn", "kr", "jp" //experimental
			}; 
	
	private static final String PEA_NAME = "File-Lock-PEA";
	private static final String VERSION = "1.6 (PEA version 2)";
	private static final String YEAR_OF_PUBLICATION = "2024"; 
	private static final String WEBSITE = "/html/file_pea.html";
	private static final String SOURCE_LINK = "https://eck.cologne/peafactory/src/downloads/filePEA/file-lock-pea-1.5-src.zip"; 

	private FileTypePanel decryptedFileTypePanel;	
	
	
	public FileControlStartWithoutFiles() {	
		pswDialog = this;
		setDialog(pswDialog);
		//dialogView = PswDialogView.getInstance();
		JREProperties.setMainWindow(PswDialogView.getInstance());
		PeaControl.setPlainTextMode(false);
	}

	public final synchronized static FileControlStartWithoutFiles getInstance() {
		if (pswDialog == null) {
			pswDialog = new FileControlStartWithoutFiles();
		} else {
			//return null
		}
		return pswDialog;
	}	

	/**
	 * start the PEA
	 * 
	 * @param args	test mode: "-t" or rescue mode: "-r"
	 */
	public static void main(String[] args) {		
		
		PeaProperties.setVersionInfo(PEA_NAME + " " + VERSION);
		PeaProperties.setFileType("file");			

		new PropertyHandler().readAndSetAllPropValues();
		
		// set working mode: -t for test mode: write stderr to log file
		if(args != null && args.length > 0) {
			//PeaProperties.setWorkingMode(args[0]);
			//TestLog.v(FileControl.class, "ARGS: " + args[0]);
			boolean runCommands = CommandLineHandler.handleArgs(args);
			TestLog.v(FileControl.class, "Run in command line mode: " + runCommands);
			
			if (runCommands == false) {
				PeaProperties.setCommandLineMode(false);
			}
		}
		if (PeaProperties.isTestMode()) {
			boolean errResult = TestLog.createErrLogFile();
			if (errResult == false) {
				PeaProperties.setWorkingMode("");
				TestLog.e(FileControl.class, "Couldn't create log file... \n Reset test mode...");
			} 
			TestLog.v(FileControl.class, PEA_NAME + " " + VERSION);
			TestLog.v(FileControl.class, System.getProperty("os.name") + "\n"
					+ "Java default:" + System.getProperty("java.version"));
			if (PeaProperties.isCommandLineMode() == true) {
				TestLog.v(FileControl.class, "Run in command line mode...");
			} else {
				TestLog.v(FileControl.class, "Start GUI...");
			}
		}		
		
		if (PeaProperties.isCommandLineMode() == false) {
			// encrypt open file if shutdown without closing the program:
			ShutdownHookThread hook = new ShutdownHookThread();
			Runtime.getRuntime().addShutdownHook(hook);
			PeaControl.initializeVariables();				

			SwingUtilities.invokeLater(new EventDispatchThread());
		} else {
			TestLog.v(FileControl.class, "Command line mode...");
		}
		//TestLog.e(FileControl.class,"------------------");
	}
	
	// indicate that the opened files are actually encrypted or plain text
	protected final static synchronized void setDecrypted(boolean _decrypted) {
		decrypted = _decrypted;
	}

	
	//-------------------HELPER-FUNCTIONS--------------		

	@Override
	public void preComputeInThread() {
		checkUpdates();
	}

	@Override
	public void clearSecretValues() {
		// keys are necessary to encrypt files later, so do not clear
	}
	
	/**
	 * Reset cursor, enable ok button and optionally
	 * update the view of the password dialog.
	 * This is called in the done method only.
	 * 
	 * @param returnToDialog	true, if the program should return to the 
	 * 							password dialog and check boxes were added or removed,
	 * 							eventually set salt and key are to be cleared
	 * 							(an error occurred)
	 */
	private void cleanUpOnReturn(boolean returnToDialog) {		
		if (PeaProperties.isCommandLineMode() == true && returnToDialog == true) {
			TestLog.ve(FileControl.class, "Decryption failed",5);
			System.out.println("Program exists...");
			System.exit(1);
		}
		
		PswDialogView.getInstance().setCursor(CursorManager.getDefaultCursor());
		if (returnToDialog == true) {
			PeaControl.setPlainTextMode(false);
			PswDialogView.enableOkButton();
			//KeyFileDialogs.resetProbablyNewDefaultKeyFileName();
			if (KeyFileDialogs.isNewChosenDefaultKeyFile() == true) {
				PeaProperties.setDefaultKeyFileName(null);
			}
			CipherStuff.getSessionKeyCrypt().clearKeys();
			// maybe files were removed...
			decryptedFileTypePanel.getFileComposer().getFileModel().resetTreeMap();
			//decryptedFileTypePanel.updateNumberAndSize();
			//TestLog.o(FileControl.class, "NumberSize: " + decryptedFileTypePanel.getFileComposer().getCurrentNumberAndSize());
			FilePanelHolder.updateFileView();
			PswDialogView.clearPassword();
			JREProperties.setMainWindow(PswDialogView.getInstance());
			//dialogView.pack();
		} else {
			if (lockFrame != null) {
				PeaControl.setPlainTextMode(true);
				JREProperties.setMainWindow(lockFrame);			
			} else {
				TestLog.ve(FileControl.class, "Missing lockFrame", 5);
			}
		}
		PswDialogView.setInitializing(false);
	}

	@Override
	public void startDecryption() {
		
		byte[] keyMaterial = null;
		boolean printErrors = false;
		if (PeaProperties.isTestMode()) {
			printErrors = true;
		}		
		if ( SessionKeyCrypt.isKeySet() == true) {
			TestLog.ve(FileControl.class, "Key is set before KDF", 5);
		}

		// create the panel, composer and model for decryptedFiles
		FileModel decryptedFileModel = new FileModel(false, null, null);
		FileComposer decryptedFileComposer = new FileComposer(decryptedFileModel);
		decryptedFileTypePanel = new FileTypePanel(300,300, true, false, decryptedFileComposer, lockFrame);
		
		// set FileTypePanel for LockFrame:
		LockFrameFile.setFileDisplayPanel(decryptedFileTypePanel);

		// create the frame: 	
		int[] dimValues = JREProperties.getFrameDim();
		Point loc = new Point(0,0);
		if (PeaProperties.isCommandLineMode() == true) {
			// get location from properties
			loc = new Point (JREProperties.getFrameLocation()[0], JREProperties.getFrameLocation()[1]);
		} else {
			loc = PswDialogView.getView().getLocation();
		}
		lockFrame = LockFrameFile.getInstance(
				loc, 
				new Dimension(dimValues[0], dimValues[1]));	

		// the selected file names to decrypt
		String[] fileNames = null;
		
		boolean startWithoutFiles = false;

		try {

			if ( PswDialogView.isInitializing() == true ){
				
				if (PeaProperties.isCommandLineMode() == true) {
					// should never happen in command line mode:
					TestLog.ve(getClass(), "Invalid behaviour: Initializing in command line mode. Program exits.");
					cleanUpOnReturn(false);
				}
				TestLog.o(FileControl.class, "Initialization");
				decryptedFileTypePanel.getFileComposer().setPlainModus(true); // files are plain text

				keyMaterial = getKeyMaterial();

				if (keyMaterial == null) { // bug
					new UnexpectedValueException("keyMaterial", "byte[]", "is null").printDescription();
					cleanUpOnReturn(true);
					return;
				}
				// store session key:
				CipherStuff.getSessionKeyCrypt().storeKey(keyMaterial, KeyDerivation.getAlgoParamsforLastKDF());

				// create new Nonce and set
				byte[] nonce = RandomStuff.createRandomBytes(Attachments.getNonceSize());
				Attachments.setNonce(nonce);

				// reset tree map from previous processes e.g. if password failed:
			//	decryptedFileTypePanel.getFileComposer().getFileModel().resetTreeMap();
				// add files
				boolean added = decryptedFileTypePanel.getFileComposer().addAction(PswDialogView.getView(),
						FilePEASetting.fileTranslate("select_file_password_initialize"), 
						true, false);//, decryptedFileTypePanel.getLocationOnScreen());

				if (added == true) {
					//TestLog.o(FileControl.class, "added");
					PswDialogView.clearPassword();			
					lockFrame.setVisible(true);		
					JREProperties.setMainWindow(lockFrame);
					PeaControl.setPlainTextMode(true);
					LockFrameFile.setFileDisplayPanel(decryptedFileTypePanel);					

					decryptedFileTypePanel.updateWindow();
					
				} else { // no file was added
					TestLog.o(FileControl.class, "no files added");
					cleanUpOnReturn(true);
					// error message
					PswDialogView.setMessage(PeaProperties.getVmBridge().translate("no_valid_file_selected"), true);
					return;
				}
				
				if (decryptedFileTypePanel.getFileComposer().getFileModel().checkForValidFile() == false){
					TestLog.e(getClass(), "no valid files found");
					
					PeaDialog.showMessage(PswDialogView.getInstance(),//PeaControl.getMainWindow(), 
							PeaProperties.getVmBridge().translate("no_valid_file_found") + "\n\n " +
							PeaProperties.getVmBridge().translate("open_file") + ": \n  - "
							+ PeaProperties.getVmBridge().translate("menu") + "\n     - "
							+ FilePEASetting.fileTranslate("add_new_files_to_encrypt")
							 ) ;
				}
				JREProperties.setTypePanel(decryptedFileTypePanel);					
				// close this view
				PswDialogView.getView().setVisible(false);	
				JREProperties.setMainWindow(lockFrame);
				LockFrameFile.setFileDisplayPanel(decryptedFileTypePanel);
				PeaControl.setPlainTextMode(true);
				// encrypt by shutdown:
				setDecrypted(true);
				//decrypted = true;
				cleanUpOnReturn(false);
				return;			
				
			} else { // not initializing

				// get files to encrypt:
				if (PeaProperties.isCommandLineMode() == false) {
					if (this.getEncryptedFileTypePanel() == null || this.getEncryptedFileTypePanel().getAllFileNames() == null) { //  no file
						// Do not start without a password or keyfile
						if (PeaProperties.getDefaultKeyFileFlag() == 0  && 
								(PswDialogView.getPassword() == null || PswDialogView.getPassword().length == 0)) {

							cleanUpOnReturn(true);
							PswDialogView.setMessage(PeaProperties.getVmBridge().translate("no_valid_file_selected"), true);
							return;
						} else {
							// File Lock PEA version 1.6: open File Lock PEA without selected files
							startWithoutFiles = true;
						}
					} else {
						fileNames = this.getEncryptedFileTypePanel().getValidLocalFileNames(true, false);
					}
				} else {
					fileNames = CommandLineHandler.getFileNames();
				}

/* change 1.6				if (fileNames == null || fileNames.length == 0) {
					TestLog.e(getClass(), "No file was selected");
					PswDialogView.setMessage(PeaProperties.getVmBridge().translate("no_valid_file_selected"), true);
					JREProperties.setMainWindow(PswDialogView.getInstance());
					//PeaControl.setPlainTextMode(false);
					//PswDialogView.enableOkButton();
					cleanUpOnReturn(true);
					return;
				} */

				// KDF:
				// derive the key from password and salt: 
				keyMaterial = getKeyMaterial();

				if (keyMaterial == null) {
					TestLog.e(getClass(), "Key derivation failed. Key material is null", 5);				
					cleanUpOnReturn(true);
					PswDialogView.setMessage("Key derivation failed", true);
					//PswDialogView.clearPassword();
					//dialogView.pack();
					JREProperties.setMainWindow(PswDialogView.getInstance());
					//PeaControl.setPlainTextMode(false);
					//PswDialogView.enableOkButton();
					return;
				} else {
					// store the key to use later:
					CipherStuff.handleKey(keyMaterial, KeyDerivation.getAlgoParamsforLastKDF(), true);
				}
			}
		} catch (Exception ex) {

			TestLog.e(getClass(), ex.toString() + " - " + ex.getMessage() + "  program exits");
			ex.printStackTrace();
			System.exit(2);
		}
		String[] errorMessages = null;

		if (PeaProperties.isCommandLineMode() == false) {
			if (startWithoutFiles == false && FilePanelHolder.getSelectedModel() != null) {// change 1.6
				FileModel encryptedFileModel = FilePanelHolder.getSelectedModel();
				//((FileTypePanel)JREProperties.getTypePanel()).getFileComposer().getFileModel();
				// cleanup the map
				encryptedFileModel.cleanupMap();// remove invalid and deselected files
				if( encryptedFileModel.getAllSize() > (16 * 1024 * 1024)
						|| encryptedFileModel.getFileNumber() > 64) {
					decryptedFileTypePanel.startProgressTask();
				}
				decryptedFileTypePanel.getFileComposer().setPlainModus(false); // files are encrypted
				if (fileNames != null) {
					decryptedFileTypePanel.getFileComposer().addFilesToMap(new ArrayList<String>(Arrays.asList(fileNames)) , false, false, false);
					//for(int i=0;i<decryptedFileModel.getAllFiles().length;i++) TestLog.o(FileControl.class, decryptedFileModel.getAllFiles()[i].getAbsolutePath());
					errorMessages = CipherStuff.getCipherMode().decryptFiles( fileNames, CipherStuff.getSessionKeyCrypt().getKey()[0],// keyMaterial,
						true, decryptedFileTypePanel.getFileComposer() );
				}
			}
		} else {
			errorMessages = CipherStuff.getCipherMode().decryptFiles( fileNames, keyMaterial,
					true, decryptedFileTypePanel.getFileComposer() );
			for (int i = 0; i < errorMessages.length; i++) {
				if (errorMessages[i] != null) {
					TestLog.ve(FileControl.class, fileNames[i] + ": " + errorMessages[i]);
					TestLog.e(getClass(), fileNames[i] + ": " + errorMessages[i]);
				}				
			}
		}

		decryptedFileTypePanel.getFileComposer().setPlainModus(true); // files are decrypted now
			
		// to check if password failed
		boolean passwordFailed = false;
		ArrayList<String> passwordFailedFiles = new ArrayList<String>();
		// set annotations from errorMessages:
		if (startWithoutFiles == false && errorMessages != null) {// change 1.6
			int len = errorMessages.length;			
			for (int i = 0; i < len; i++) {
				String message = errorMessages[i];
				if (message != null) {
					if (printErrors == true){
						TestLog.e(getClass(), fileNames[i] + ": " + message);
					}
					if (message.equals(PeaProperties.getVmBridge().translate("data_integrity_violated"))) {
						PeaDialog.showMessage(JREProperties.getMainWindow(), 
								PeaProperties.getVmBridge().translate("file") + ": " + fileNames[i] + "\n"
										+ PeaProperties.getVmBridge().translate("integrity_check_failed_message"), 
										PeaProperties.getVmBridge().translate("error"), 0);
						// The file was decrypted
						// add to valid files but with annotation: 
						decryptedFileModel.setAnnotation(
								new File(fileNames[i]), PeaProperties.getVmBridge().translate("data_integrity_violated") + ": ");//"");					
					} else {
						if (message.equals(PeaProperties.getVmBridge().translate("password_failed"))){
							passwordFailedFiles.add(fileNames[i]);
							passwordFailed = true;
						} 
						decryptedFileModel.setAnnotation(
								new File(fileNames[i]), 
								message + FileModel.getInvalidMarker());
					}
				} else {
					// file is decrypted - no invalid marker, but annotation
					decryptedFileModel.setAnnotation(
							new File(fileNames[i]), "");
				}
			}
		}

		// close progress task
		decryptedFileTypePanel.closeProgressTask();

		if (startWithoutFiles == false) { // version 1.6
			// check for valid file:
			if (decryptedFileModel.checkForValidFile() == false) {
				if (PeaProperties.isCommandLineMode() == true) {
					if (passwordFailed == true) {
						System.out.println(PeaProperties.getVmBridge().translate("password_failed"));
					} else {
						System.out.println(PeaProperties.getVmBridge().translate("decryption_failed"));
					}
				} else {
					if (passwordFailed == true) {
if (fileNames == null) {
	PswDialogView.setMessage(PeaProperties.getVmBridge().translate("no_valid_file_selected"), true);
} else {
						PswDialogView.setMessage(
								PeaProperties.getVmBridge().translate("password_failed"), true);
}
					} else {
						PswDialogView.setMessage(PeaProperties.getVmBridge().translate("decryption_failed"), true);
					}
					TestLog.e(getClass(), "password failed for all files");
					decryptedFileModel.resetTreeMap();
					if(lockFrame.isValid()){
						lockFrame.setVisible(false);
						JREProperties.setMainWindow(PswDialogView.getInstance());
						//PeaControl.setPlainTextMode(false);
					}
				}
				cleanUpOnReturn(true);

				//PswDialogView.enableOkButton();
				JREProperties.setMainWindow(PswDialogView.getInstance());
				//PeaControl.setPlainTextMode(false);
				return;
			} else {
				if (passwordFailedFiles.isEmpty() == false) {
					// show password failed files:
					StringBuilder failedFilesString = new StringBuilder();
					for (String s : passwordFailedFiles) {
						failedFilesString.append(s);
						failedFilesString.append("\n");
					}
					PeaDialog.showMultiLineMessage(JREProperties.getMainWindow(), 
							PeaProperties.getVmBridge().translate("password_failed") + ":\n\n"
									+ failedFilesString, 
									PeaProperties.getVmBridge().translate("password_failed"), 0);
				}
			}
		}
		cleanUpOnReturn(false);
		TestLog.o(FileControl.class, "file(s) decrypted");
		// encrypt by shutdown:
		setDecrypted(true);
		//decrypted = true;
		JREProperties.setTypePanel(decryptedFileTypePanel);

		if (PeaProperties.isCommandLineMode() == false) {
			PswDialogView.clearPassword();			
		}
		decryptedFileTypePanel.getFileComposer().updateFiles(false);
			
		JREProperties.setMainWindow(lockFrame);
		PeaControl.setPlainTextMode(true);

		decryptedFileTypePanel.updateWindow();
		LockFrameFile.setFileDisplayPanel(decryptedFileTypePanel);
		
		if (PeaProperties.isCommandLineMode() == true) {
			// number and size is not calculated in add action
			decryptedFileTypePanel.getFileComposer().getFileModel().calculateNumberAndSize(decryptedFileTypePanel.getValidLocalFileNames(true, false));
			decryptedFileTypePanel.updateNumberAndSize();
		}
		
		//decryptedFileModel.calculateNumberAndSize(decryptedFileTypePanel.getValidLocalFileNames(true, false));	
		lockFrame.setVisible(true);	
		
		// set property to register interrupted closing 
		// and show invalidFiles 
		FilePEASetting.setClosedProperly(false);
		FilePEASetting.setShowInvalidPanel(true);
		new PropertyHandler().updateSpecificPeaPropValues();
		enableWarningOnOS();
		
		if (PeaProperties.getTimeout() != null && PeaProperties.getTimeout().length() > 0) {
			// set timer:
			TimeoutTimer.startInstance(PeaProperties.getTimeout(), lockFrame);
		}
		
		if (PeaProperties.isCommandLineMode() == false) {
			PswDialogView.getView().setVisible(false);	
		}
		
		if (PeaProperties.isCommandLineMode() == true) { // does not show file number and size otherwise
			lockFrame.resetFrame();
		}
		if (decryptedFileTypePanel.getFileComposer().getFileModel().getAllFiles() ==  null
				|| decryptedFileTypePanel.getFileComposer().getFileModel().getAllFiles().length == 0) {
			
			decryptedFileTypePanel.getFileComposer().addAction(JREProperties.getMainWindow(),
					PeaProperties.getVmBridge().translate("select_files_to_encrypt"), 
					true, true);	// hidden file option = true, check again, = true
			// adding with checkAgain=true results in wrong fileNumber:
			decryptedFileTypePanel.getFileComposer().getFileModel().calculateNumberAndSize(decryptedFileTypePanel.getValidLocalFileNames(true, false));
			decryptedFileTypePanel.updateNumberAndSize();

			//messageLabel.setText("");
		}
	}
	

	// used in PswDialogView to add selected file to path file:
	@Override
	public String[] getSelectedFileNames() {
		return FilePanelHolder.getSelectedPanel().getValidLocalFileNames(true, false);
	}

	@Override
	public String initializeNewFiles() {
		// get default values from properties
		String algoParamString = PeaProperties.getDefaultAlgoParamString();//new String(PeaProperties.getAlgoParamString(), AttachmentHandler.getASCIICharset());
		TestLog.v(FileControl.class, "Default algorithm-parameter string: " + algoParamString);
		// set to use these settings
		String readError = AlgoParamHandler.readString(algoParamString, false);
		// set algo parameter string to use
		PeaProperties.setAlgoParamString(algoParamString.getBytes(AttachmentHandler.getASCIICharset()));//AlgoParamHandler.writeString().getBytes(AttachmentHandler.getASCIICharset()));		
		// The rest is done in startDecryption with initializing == true
		return readError;
	}

	@Override
	public void openEncryptedFiles() {
		
		boolean added = AttachmentHandler.addEncryptedFiles(JREProperties.getMainWindow(), 
				PeaProperties.getVmBridge().translate("open_encrypted_file"), 
				true, true);// hidden files, do not check listed files

		if (added == false) { // no file was added
			TestLog.e(FileControl.class, "no file was added");
			PswDialogView.setMessage(PeaProperties.getVmBridge().translate("no_valid_file_found"), true);
			//PeaDialog.showMessage(JREProperties.getMainWindow(), 
			//		PeaProperties.getVmBridge().translate("no_valid_file_found"), null, -1);
			return;
		} else {
			TestLog.v(FileControl.class, "File added");
		}
		
		boolean validFileFound = FilePanelHolder.getSelectedModel().checkForValidFile();
		if (validFileFound == false) {
			TestLog.e(FileControl.class, "No valid file found");
			//PswDialogView.setMessage(PeaProperties.getOsBridge().translate("no_valid_file_found"), true);
			PswDialogView.setMessage(PeaProperties.getVmBridge().translate("open_or_initialize"), true);
		} else {
			// clear errorMessageLabel
			PswDialogView.setMessage("   ", true);
		}	
		FilePanelHolder.updateWindow();
	}
	
	@Override
	public final LockFrameFile getLockFrame() {
		return lockFrame;
	}
	@Override
	public void setLockFrame(PeaLockFrame _lockFrame) {
		if (_lockFrame instanceof LockFrameFile) {
			this.lockFrame = (LockFrameFile) _lockFrame;
			if (JREProperties.getMainWindow() instanceof PeaLockFrame) {
				JREProperties.setMainWindow(this.lockFrame);
			}
		} else {
			TestLog.e(FileControl.class, "unexpected error");
			TestLog.ve(FileControl.class, "Invalid frame...", 5);
		}
	}
	@Override
	public String getVersion() {
		return VERSION;
	}
	@Override
	public String getYearOfPublication() {
		return YEAR_OF_PUBLICATION;
	}
	@Override
	public String getWebsite() {
		return WEBSITE;
	}
	@Override
	public String getSourceLink() {
		return SOURCE_LINK;
	}

	@Override
	public boolean checkFormat(File file, boolean easyCheck, boolean middleCheck, boolean expensiveCheck) {
		// nothing to check: all formats are accepted
		return true;
	}
	@Override
	protected byte[] getPlainBytes() {
		// not used
		return null;
	}

	@Override
	public String setPeaPropValues(Properties prop, InputStream inputStream) {
		TestLog.v(getClass(), "set specific properties.");
		String errorMessage = null; 
		try {
			
			String properlyClosed = prop.getProperty("properly_closed");
			if (properlyClosed != null) {
				if (properlyClosed.toUpperCase().equals("TRUE")) {
					FilePEASetting.setClosedProperly(true);
				} else if (properlyClosed.toUpperCase().equals("FALSE")) {
					FilePEASetting.setClosedProperly(false);
				} else {
					TestLog.ve(FileControl.class, "invalid value for properly_closed: " + properlyClosed);
				}
			}
			
			String showInvalidPanel = prop.getProperty("show_invalid_files");
			if (showInvalidPanel != null) {
				if (showInvalidPanel.toUpperCase().equals("TRUE")) {
					FilePEASetting.setShowInvalidPanel(true);
				} else if (showInvalidPanel.toUpperCase().equals("FALSE")) {
					FilePEASetting.setShowInvalidPanel(false);
				} else {
					TestLog.ve(FileControl.class, "invalid value for show_invalid_files: " + showInvalidPanel);
				}
			}
						
		} catch ( Exception e){
			TestLog.e(FileControl.class, e + ", " + e.getMessage());
			new UnexpectedValueException("","","set pea properties").printDescription();
			errorMessage = e.toString() + ", " + e.getLocalizedMessage();
		}
		return errorMessage;		
	}

	@Override
	public String updatePeaPropValues(Properties prop) {//, InputStream inputStream) {
		String errorMessage = null; 
		try {
			
			boolean properlyClosed = FilePEASetting.isClosedProperly();
			if (properlyClosed == true) {
				prop.setProperty("properly_closed", "TRUE");
			} else {
				prop.setProperty("properly_closed", "FALSE");
			}
			
			boolean showInvalidPanel = FilePEASetting.isShowInvalidPanel();
			if (showInvalidPanel == true) {
				prop.setProperty("show_invalid_files", "TRUE");
			} else {
				prop.setProperty("show_invalid_files", "FALSE");
			}
			
		} catch ( NullPointerException e){
			TestLog.e(FileControl.class, new ExceptionHelper(e).getInfos());
			errorMessage = e.toString() + ", " + e.getLocalizedMessage();
		} catch ( Exception e){
			TestLog.e(FileControl.class, new ExceptionHelper(e).getInfos());
			errorMessage = e.toString() + ", " + e.getLocalizedMessage();
		}
		return errorMessage;
	}
	
	//============ helper functions =============
	

	private final void enableWarningOnOS() {
		// check OS: Linux/Unix or Windows
		
		// Linux: 
	/*	try {
			if ( ! new File(System.getProperty("user.dir") + File.separator + "resources" + File.separator + "warning-unencrypted-files-left.sh").exists() ) {
				// copy shell script in folder resources
				Files.copy(getClass().getResourceAsStream("/resources/warning-unencrypted-files-left.sh"), 
					Paths.get(System.getProperty("user.dir") + File.separator + "resources") );
				// make shell script executable
				new File(System.getProperty("user.dir") + File.separator + "resources" + File.separator + "warning-unencrypted-files-left.sh").setExecutable(true);
			}
		} catch (IOException e) {
			TestLog.e(getClass(), "Can not enable warning mechanism for unproperly closing: " + e.getMessage());
		}		
		// execute crontab
		Runtime rt = Runtime.getRuntime();
		try {
			rt.exec("crontab -e; @reboot " + System.getProperty("user.dir") + File.separator + "resources" + File.separator + "warning-unencrypted-files-left.sh");
			rt.exec("crontab -l 2>/dev/null; echo @reboot /media/ALL/prog/eclipse-peafactory/peafactory/resources/File/warning-unencrypted-files-left.sh\") | crontab -");
			
			rt.exec("(crontab -u ax -l ; echo \'@reboot sleep 1 && /home/ax/warning-unencrypted-files-left.sh\') | crontab -u ax -");
					
		} catch (IOException e) {
			TestLog.e(getClass(), "Can not enable warning mechanism for unproperly closing (create crontab): " + e.getMessage());
		}
		*/
		// Windows: create script in autostart folder
		
	}
	protected final void disableWarningOnOS() {
		// check OS: Linux/Unix or Windows
		
		// Linux: stop crontab
		
		// Windows: delete script in autostart folder
	}
	
	//============ inner classes ================
	
	private static class  EventDispatchThread implements Runnable{

		@Override
		public void run() {
			
			pswDialog = FileControlStartWithoutFiles.getInstance();	
			PswDialogView.getInstance().setVisible(true);
			JREProperties.setMainWindow(PswDialogView.getInstance());
			setDialog(pswDialog);
			startEDT();
			
			// read properties from resource file and set:
			new PropertyHandler().setSpecificPropValues();
			
			if (FilePEASetting.isShowInvalidPanel() == true) { 
				FilePanelHolder.showInvalidPanel();
			}
			
			if (FilePEASetting.isClosedProperly() == false) { 
				TestLog.e(getClass(), "The last session was not closed properly. \n" 
						+ "Some files may not have been re-encrypted. \n");
				PeaDialog.showMessage(JREProperties.getMainWindow(), 				
						FilePEASetting.fileTranslate("warning_improper_closing") + "\n" 
						+ FilePEASetting.fileTranslate("warning_no_re-encryption") + "\n"
						+ FilePEASetting.fileTranslate("see_invalid_file_list") + "\n",
					//	+ "You may find these files in the list of invalid files. ", 
						PeaProperties.getVmBridge().translate("warning"), 0);				
			}
			
			if (FilePanelHolder.getFileNumber() == 0) {
				PswDialogView.setMessage(PeaProperties.getVmBridge().translate("open_or_initialize"), true);
			} else {				
				FilePanelHolder.updateWindow();
			}
		}		
	}
	
	/*
	 * Try to encrypt the opened files if  before the shutdown
	 */
	private static class ShutdownHookThread extends Thread {

		public void run() {
			
			//TestLog.e(getClass(), "shutdownhook");
			if (System.getProperty("os.name").contains("BSD")){
				// FreeBSD destroys the files in ShutdownHook...
				CipherStuff.getSessionKeyCrypt().clearKeys();
			} else {
				if (decrypted == true) {
					if (LockFrameFile.getFileDisplayPanel().
							getFileComposer().getFileModel().getAllSize() 
							< (1024 * 4)) {// < 4 KB
						String errorMessage = pswDialog.getLockFrame().encryptCurrentFiles(null);
						if (errorMessage != null) {
							PeaDialog.showMessage(JREProperties.getMainWindow(), 
									errorMessage, PeaProperties.getVmBridge().translate("error"), 0);
						}
						CipherStuff.getSessionKeyCrypt().clearKeys();
						decrypted = false;
					} else {
						// do not encrypt files: the execution time is long,
						// there is a risk of half-encrypting (damaging) the files
						CipherStuff.getSessionKeyCrypt().clearKeys();
					}
				}
			}
		}
	}

	@Override
	public String getPeaName() {
		return PEA_NAME;
	}

	@Override
	public String[] getSupportedExtensions() {
		return null;
	}

	@Override
	public void setLanguage(String languageShortName) {
		// there is no extra bundle	
	}

	@Override
	public String getPeaPropFileName() {
		return peaPropFileName;
	}

	@Override
	public String[] getSupportedLanguages() {
		return supportedLanguages;
	}

	@Override
	public String translatePeaSpecific(String key) {
		// TODO HelpMenu instruction
		return "";
	}
}	