package cologne.eck.file_pea;

/*
 * Peafactory - Production of Password Encryption Archives
 * Copyright (C) 2025  Axel von dem Bruch
 * 
 * This library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published 
 * by the Free Software Foundation; either version 2 of the License, 
 * or (at your option) any later version.
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE. 
 * See the GNU General Public License for more details.
 * See:  http://www.gnu.org/licenses/gpl-2.0.html
 * You should have received a copy of the GNU General Public License 
 * along with this library.
 */


import java.awt.*;
import java.awt.event.*;
import java.io.File;
import java.util.ArrayList;
import java.util.HashSet;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.border.LineBorder;

import cologne.eck.all_peas.cloud.gui.CloudMenu;
import cologne.eck.all_peas.control.PathFileManager;
import cologne.eck.all_peas.control.PeaControl;
import cologne.eck.all_peas.data.PeaProperties;
import cologne.eck.all_peas.files.ExecutionTimeObserver;
import cologne.eck.all_peas.files.FileModel;
import cologne.eck.all_peas.files.FileTypePanel;
import cologne.eck.all_peas.gui.CursorManager;
import cologne.eck.all_peas.gui.IconManager;
import cologne.eck.all_peas.gui.PeaDialog;
import cologne.eck.all_peas.gui.PeaFileChooser;
import cologne.eck.all_peas.gui.PeaLockFrame;
import cologne.eck.all_peas.gui.PswDialogView;
import cologne.eck.all_peas.gui.TimeoutSetter;
import cologne.eck.all_peas.gui.menu.HelpMenu;
import cologne.eck.all_peas.gui.menu.PeaMenuBar;
import cologne.eck.all_peas.gui.menu.SettingMenu;
import cologne.eck.all_peas.gui.menu.ViewMenu;
import cologne.eck.all_peas.vm_specific.JREProperties;
import cologne.eck.peafactory.crypto.CipherStuff;
import cologne.eck.tools.Converter;
import cologne.eck.tools.FileDropHandler;
import cologne.eck.tools.FileTools;
import cologne.eck.tools.PropertyHandler;
import cologne.eck.tools.TestLog;
import cologne.eck.tools.UnexpectedValueException;
import cologne.eck.tools.ZipStuff;



@SuppressWarnings("serial")
public
class LockFrameFile extends PeaLockFrame 
		implements ActionListener,  WindowListener {

	private static volatile LockFrameFile frame;// static instance
	private static FileTypePanel filePanel;// associated FileTypPanel to show the files

	private JButton timeoutButton = null;
	
	private static JLabel messageLabel;
	
	private static boolean timerStarted = false;

	/**
	 * Main frame to display the selected files for file pea. 
	 * 
	 * @param location	location on screen
	 * @param dim		dimension
	 */
	private LockFrameFile(Point location, Dimension dim){
		
		boolean previousMode = PeaControl.isPlainTextMode();
		// always plain text mode - CloudMenu needs this
		PeaControl.setPlainTextMode(true);
		
		frame = this;		
		//for(int i=0;i<selectedFileNames.length;i++) TestLog.o(LockFrameFile.class, "selected: " + selectedFileNames[i]);
		
	    this.setIconImages(IconManager.getIconList());
		this.setDefaultCloseOperation(DO_NOTHING_ON_CLOSE); // do not allow to close or to exit:
															// files may remain unencrypted
		this.addWindowListener(this);// for windowClosing
		
		PswDialogView pdv = PswDialogView.getInstance();
		
		JPanel contentPane = new JPanel();
		contentPane.setLayout(new BoxLayout(contentPane, BoxLayout.PAGE_AXIS));
		this.setContentPane(contentPane);				
		
		JPanel menuPanel = new JPanel();
		menuPanel.setLayout(new BoxLayout(menuPanel, BoxLayout.X_AXIS));
		menuPanel.setMinimumSize(new Dimension(200, 30));
		menuPanel.setMaximumSize(new Dimension((int) PswDialogView.getScreenwidth(), 40));
		
		PeaMenuBar menuBar = new PeaMenuBar();
		//menuBar.setBorder(new EtchedBorder() );
		
		Font font = new Font(Font.SANS_SERIF, Font.PLAIN, PeaProperties.getFontSize());
		
		JMenu menu = new JMenu();
		menu.getAccessibleContext().
	    setAccessibleDescription(
	    	    "Clicking this component opens a menu for file operations.");
		menu.setFont(font);
		menu.setText(PeaProperties.getVmBridge().translate("menu"));//"Menu");
		menu.setMnemonic(pdv.getMnemo(menuBar, PeaProperties.getVmBridge().translate("menu")));
		menuBar.add(menu);
		
		JMenuItem encryptItem = new JMenuItem();
		encryptItem.getAccessibleContext().
	    setAccessibleDescription(
	    	    "Clicking this component encrypts and closes all selected files and exits.");
		encryptItem.setText(FilePEASetting.fileTranslate("close_and_encrypt_selected_files"));//"close and encrypt selected files");
		encryptItem.setActionCommand("encryptWithClosing");
		encryptItem.setFont(font);
		//encryptItem.setMnemonic(KeyEvent.VK_E);
		//encryptItem.setMnemonic(pdv.getMnemo(menu, FilePEASetting.fileTranslate("close_and_encrypt_selected_files")));
		encryptItem.addActionListener(this);
		menu.add(encryptItem);
		
		JMenuItem unencryptItem = new JMenuItem();
		unencryptItem.getAccessibleContext().
	    setAccessibleDescription(
	    	    "Clicking this component leaves selected files unencrypted and closes the program.");
		unencryptItem.setText(FilePEASetting.fileTranslate("close_and_leave_unencrypted"));//"close and leave all opened file unencrypted");
		unencryptItem.setActionCommand("leaveUnencrypted");
		unencryptItem.setFont(font);
		//unencryptItem.setMnemonic(pdv.getMnemo(menu, FilePEASetting.fileTranslate("close_and_leave_unencrypted")));
		unencryptItem.addActionListener(this);
		menu.add(unencryptItem);
		
		JMenuItem addItem = new JMenuItem();
		addItem.getAccessibleContext().
	    setAccessibleDescription(
	    	    "Clicking this component opens a dialog to add new plain text files.");
		addItem.setText(FilePEASetting.fileTranslate("add_new_files_to_encrypt"));//"add new files to encrypt");
		addItem.setActionCommand("addNewFiles");
		addItem.setFont(font);
		//addItem.setMnemonic(pdv.getMnemo(menu, FilePEASetting.fileTranslate("add_new_files_to_encrypt")));
		addItem.addActionListener(this);
		menu.add(addItem);
		
		JMenuItem openItem = new JMenuItem(PeaProperties.getVmBridge().translate("open_encrypted_file"));//CalendarSetting.calTranslate("instruction"));
		openItem.getAccessibleContext().
	    setAccessibleDescription(
	    "Clicking this component opens a dialog to open an encrypted file.");
		openItem.setActionCommand("openEncrypted");
		openItem.setFont(font);
		openItem.addActionListener(this);
		menu.add(openItem);

		JMenuItem quitItem = new JMenuItem();
		quitItem.getAccessibleContext().
	    setAccessibleDescription(
	    	    "Clicking this component exits the program.");
		quitItem.setText(PeaProperties.getVmBridge().translate("quit"));//"change password");
		quitItem.setActionCommand("quit");
		quitItem.setFont(font);
		//quitItem.setMnemonic(pdv.getMnemo(menu, PeaProperties.getVmBridge().translate("quit")));
		quitItem.addActionListener(this);
		menu.add(quitItem);
		
		JMenu viewMenu = new ViewMenu(null, filePanel, 
				pdv.getMnemo(menuBar, PeaProperties.getVmBridge().translate("view")), true);// plainModus true
		menuBar.add(viewMenu);
		
		JMenu extraMenu = new JMenu();
		extraMenu.getAccessibleContext().
	    setAccessibleDescription(
	    	    "Clicking this component opens a menu for zip and wipe operations.");
		extraMenu.setFont(font);
		extraMenu.setText(PeaProperties.getVmBridge().translate("extra"));//"Menu");
		extraMenu.setMnemonic(pdv.getMnemo(menuBar, PeaProperties.getVmBridge().translate("extra")));
		menuBar.add(extraMenu);
		
		JMenuItem zipItem = new JMenuItem();
		zipItem.getAccessibleContext().
	    setAccessibleDescription(
	    	    "Clicking this component zippes the selected files.");
		zipItem.setText(FilePEASetting.fileTranslate("zip_and_add_selected_files"));
		zipItem.setActionCommand("zipSelection");
		zipItem.setFont(font);
		//zipItem.setMnemonic(pdv.getMnemo(extraMenu, FilePEASetting.fileTranslate("zip_and_add_selected_files")));
		zipItem.addActionListener(this);
		extraMenu.add(zipItem);
		
		JMenuItem unzipItem = new JMenuItem();
		unzipItem.getAccessibleContext().
	    setAccessibleDescription(
	    	    "Clicking this component unzips an archive and opens a menu to choose the folder to store.");
		unzipItem.setText(FilePEASetting.fileTranslate("choose_folder_to_zip"));;
		unzipItem.setActionCommand("zipFolder");
		unzipItem.setFont(font);
		//unzipItem.setMnemonic(pdv.getMnemo(extraMenu, FilePEASetting.fileTranslate("choose_folder_to_zip")));
		unzipItem.addActionListener(this);
		extraMenu.add(unzipItem);		
		
		JMenuItem wipeItem = new JMenuItem();
		wipeItem.setToolTipText("Overwrites file with 0, 1 and random data before deletion");
		wipeItem.setText(PeaProperties.getVmBridge().translate("wipe_and_delete"));
		wipeItem.setActionCommand("wipe");
		wipeItem.setFont(font);
		//wipeItem.setMnemonic(pdv.getMnemo(extraMenu, PeaProperties.getVmBridge().translate("wipe_and_delete")));
		wipeItem.addActionListener(this);
		extraMenu.add(wipeItem);
		
		SettingMenu setMenu = new SettingMenu(false, pdv.getMnemo(menuBar, PeaProperties.getVmBridge().translate("settings")));// decrpyted mode
		//setMenu.setMnemonic(pdv.getMnemo(menuBar, PeaProperties.getVmBridge().translate("settings")));
		menuBar.add(setMenu);
		
		CloudMenu cloudMenu = new CloudMenu(pdv.getMnemo(menuBar, PeaProperties.getVmBridge().translate("cloud")));
		//cloudMenu.setMnemonic(pdv.getMnemo(menuBar, PeaProperties.getVmBridge().translate("cloud")));
		menuBar.add(cloudMenu);
		
		JMenu helpMenu = new HelpMenu(pdv.getMnemo(menuBar, PeaProperties.getVmBridge().translate("help")));
		//helpMenu.setMnemonic(pdv.getMnemo(menuBar, PeaProperties.getVmBridge().translate("help")));
		menuBar.add(helpMenu);

		menuPanel.add(menuBar);
		menuPanel.add(Box.createHorizontalGlue());
		contentPane.add(menuPanel);
		
		contentPane.add(Box.createVerticalStrut(5) );
		
		JPanel messageLabelPanel = new JPanel();
		messageLabelPanel.setLayout(new BoxLayout(messageLabelPanel, BoxLayout.LINE_AXIS));
		
		JPanel timeoutButtonPanel = new JPanel();
		timeoutButtonPanel.setLayout(new BoxLayout(timeoutButtonPanel, BoxLayout.LINE_AXIS));
		timeoutButton = new JButton(PeaProperties.getVmBridge().translate("timer"));
		timeoutButton.setMnemonic(pdv.getMnemo(menuBar, PeaProperties.getVmBridge().translate("timer")));
		timeoutButton.addActionListener(this);
		timeoutButton.setActionCommand("setTimeout");
		timeoutButton.setBorder(new LineBorder(Color.LIGHT_GRAY));
		timeoutButton.setToolTipText(PeaProperties.getVmBridge().translate("timer_tooltip"));
		timeoutButton.setFont(new Font(Font.SANS_SERIF, Font.PLAIN, 12));
		timeoutButton.setContentAreaFilled(false);
		timeoutButtonPanel.add(new JLabel("  "));
		timeoutButtonPanel.add(timeoutButton);
		timeoutButtonPanel.add(new JLabel("  "));
		
		messageLabelPanel.add(timeoutButtonPanel);
		
		messageLabel = new JLabel("  ");
		messageLabelPanel.add(messageLabel);
		messageLabelPanel.add(Box.createHorizontalGlue());
		//messageLabel.setForeground(Color.GREEN);
		contentPane.add(messageLabelPanel);
		
		contentPane.add(Box.createVerticalStrut(20) );
		
		//
		// displayed Files:
		//
		filePanel.removeAddButton();// add button is not required because of the menu
		//filePanel.update(filePanel.getGraphics());
		// otherwise scroll bar disappears...
		filePanel.updateFileView();
		filePanel.repaint();
		filePanel.setTransferHandler(new FileDropHandler());
		contentPane.add(filePanel);		
		contentPane.add(Box.createVerticalStrut(10) );		
		
		JPanel noteLabelPanel = new JPanel();
		noteLabelPanel.setLayout(new BoxLayout(noteLabelPanel, BoxLayout.X_AXIS));
		JLabel noteLabel = new JLabel();
		noteLabel.setText(FilePEASetting.fileTranslate("selected_files_automatically_encrypted"));
		noteLabel.setFont(new Font(Font.SANS_SERIF, Font.ITALIC, 12));
		noteLabelPanel.add(noteLabel);
		noteLabelPanel.add(Box.createHorizontalGlue());
		contentPane.add(noteLabelPanel);
		
		this.setLocation(location);
		this.setPreferredSize(dim);
		
		// reset mode:
		PeaControl.setPlainTextMode(previousMode);
	}

	/*
	 * Only one static instance allowed
	 */
	protected final static LockFrameFile getInstance(Point loc, Dimension dim){

		// must create a new instance if password failed
		frame = new LockFrameFile(loc, dim);// selectedFileNames);	
		return frame;
	}
	@Override
	public final void resetFrame() {
		Point loc = frame.getLocationOnScreen();
		Dimension dim = frame.getSize();
		frame.dispose();
		frame = new LockFrameFile(loc, dim);
		frame.setSize(dim);
		frame.setVisible(true);
		((FileControl) PeaControl.getDialog()).setLockFrame(frame);
		filePanel.resetLanguage();
		if (timerStarted == true) {
			TimeoutSetter.setLockFrame(frame);
		}
		JREProperties.setMainWindow(frame);
	}

	@Override
	public void actionPerformed(ActionEvent ape) {
		String command = ape.getActionCommand();
		
		if ( command.equals("encryptWithClosing") ) {	

			closeProgram();
			
		} else if ( command.equals("leaveUnencrypted") ) {		
			messageLabel.setForeground(Color.BLACK);
			messageLabel.setText("... " + PeaProperties.getVmBridge().translate("encryption") + " " +
					PeaProperties.getVmBridge().translate("in_progress") + "... ");
			messageLabel.repaint();
			messageLabel.getParent().revalidate();
			messageLabel.getParent().repaint();
			// display a warning and an option to break
			int result = PeaDialog.showQuestion(this, 
					PeaProperties.getVmBridge().translate("leave_files_unencrypted"), 
					PeaProperties.getVmBridge().translate("warning"), 0);
			if (result == 0) {	//yes = 0, 

				CipherStuff.getSessionKeyCrypt().clearKeys();	
				FileControl.setDecrypted(false);
				System.exit(0);
			} else { // no = 1, cancel = 2
				messageLabel.setText("");
				return;
			}			
			
		} else if ( command.equals("addNewFiles") ) {	
			
			try {
				messageLabel.setForeground(Color.BLACK);
				messageLabel.setText("... " + PeaProperties.getVmBridge().translate("open_file") + " " +
						PeaProperties.getVmBridge().translate("in_progress") + "... ");

			} finally {
				//FileComposer composer = filePanel.getFileComposer();
				filePanel.getFileComposer().addAction(this,
						PeaProperties.getVmBridge().translate("select_files_to_encrypt"), 
						true, true);	// hidden file option = true, check again, = true
				// adding with checkAgain=true results in wrong fileNumber:
				filePanel.getFileComposer().getFileModel().calculateNumberAndSize(filePanel.getValidLocalFileNames(true, false));
				filePanel.updateNumberAndSize();

				messageLabel.setText("");
			}
			
		} else if (command.equals("openEncrypted")) {	

			File file = openFile(null);
			if (file != null) {				
				decryptAndLoadFileToView(file.getAbsolutePath());				
			}

		} else if ( command.equals("quit") ) {	
			// Can always close the program even if exceptions occur
			try {
				this.closeProgram();
			} catch (Exception e) {
				int result = PeaDialog.showQuestion(this, e.getLocalizedMessage() 
						+ "\n   " + PeaProperties.getVmBridge().translate("quit") + "?", 
						PeaProperties.getVmBridge().translate("error"), 0);
				if (result == 0) {
					System.exit(0);
				}
			}

		} else if ( command.equals("zipSelection") ) {	
			//TestLog.o(LockFrameFile.class, "zip selection");			
			
			// get selected file names, if no, break with message
			String[] fileNames = filePanel.getValidLocalFileNames(true, false);
			if (fileNames == null || fileNames.length == 0) {
				PeaDialog.showMessage(this, PeaProperties.getVmBridge().translate("no_valid_file_selected"), "Zip", 0);
				return;
			}
			if (FileTools.getNumberAndSize(fileNames)[1] > 1024*1024*128) { // > 256MB
				if (ExecutionTimeObserver.showWarningAndBreakOption(
			    		this, PeaProperties.getVmBridge().translate("size_warning"), false) 
						== false) {
					return;
				}
			}
			
			// get name of zip file to create, if no, break with message
			String zipName = null;
			zipName = PeaDialog.showInputDialog(this, FilePEASetting.fileTranslate("name_of_zip"), "Zip", -1);
			
			if (zipName == null || zipName.equals("")) {
				PeaDialog.showMessage(this, "Zip name is required", "Zip", 0);
				return;
			} else {			
				zipName = zipName + "_pea.zip";
				String newZipName = zipName.replaceAll("[^a-zA-Z0-9\\.\\-]", "_");
				if (! newZipName.equals(zipName)) {
					PeaDialog.showMessage(this, "Problematic characters for file name were replaced: \n"
							+ newZipName, null, 0);
					zipName = newZipName;
				}				
			}
			
			// select directory to store zip file
			File zipFolder;
			PeaFileChooser chooser = new PeaFileChooser(this);
			chooser.customizeFileChooser("Zip", FilePEASetting.fileTranslate("select_folder_to_store_zip"), false);
			chooser.setFileSelectionMode( JFileChooser.DIRECTORIES_ONLY);
			//In response to a button click:
			int returnVal = -1;
			try {
				returnVal = chooser.showOpenDialog(this);// sometimes Java bug IndexOutOfBoundsException: Invalid index
			} catch (Exception e){
				//chooser.setCurrentDirectory(new File(System.getProperty("user.home")));
				// try again
				chooser = new PeaFileChooser(this);
				chooser.customizeFileChooser("Zip", FilePEASetting.fileTranslate("select_folder_to_store_zip"), false);
				chooser.setFileSelectionMode( JFileChooser.DIRECTORIES_ONLY);
				returnVal = chooser.showOpenDialog(this);
			}
			if (chooser.getSelectedFile() == null || returnVal != JFileChooser.APPROVE_OPTION) {
				PeaDialog.showMessage(this, "You must choose a folder", "Zip", 0);
				return;					
			} else {
				zipFolder = chooser.getSelectedFile();
			}

			// zip
			String errorMessage = null;
			try {//  Wait cursor is also set in ZipStuff
				messageLabel.setForeground(Color.BLACK);
				messageLabel.setText("... " + PeaProperties.getVmBridge().translate("zip") + " " +
						PeaProperties.getVmBridge().translate("in_progress") + "...");
				// set the wait cursor: 
				frame.setCursor(CursorManager.getWaitCursor());
			} finally {

				errorMessage = ZipStuff.zipFiles(fileNames, zipName, 
						zipFolder);	
				frame.setCursor(CursorManager.getDefaultCursor());
				messageLabel.setText("");
			}

			if (errorMessage != null) {
				PeaDialog.showMessage(this, errorMessage, "Zip", 0);
				return;
			} else {
				TestLog.o(LockFrameFile.class, "Zip file created: " + zipFolder + File.separator + zipName);
			}
			int len = fileNames.length;
			
			FileModel model = filePanel.getFileComposer().getFileModel();
			// delete original files?
			String fileNamesString = "";
			for (int i = 0; i < len; i++) {
				fileNamesString += fileNames[i] + "\n";
			}
			int deleteFiles = PeaDialog.showQuestionWithScollPane(
					getFileDisplayPanel(), 
					FilePEASetting.fileTranslate("zip_created_delete_original") + "\n" + fileNamesString,
					"Zip", 0, 3);

			if (deleteFiles == 0) { // delete ok
				String errorMessageWipe = "";
				try {
					messageLabel.setForeground(Color.BLACK);
					messageLabel.setText("... " + PeaProperties.getVmBridge().translate("wipe_and_delete") + " " +
							PeaProperties.getVmBridge().translate("in_progress") + "... ");
					// set the wait cursor: 
					frame.setCursor(CursorManager.getWaitCursor());

				} finally {
					// list directories: if they become empty, delete
					ArrayList<String> dirToDelete = new ArrayList<String>();
					for (int i = 0; i < len; i++) {
						long size = FileTools.getFileSize(fileNames[i]);
						String newError = FileTools.wipeFile(fileNames[i], true);
						if (newError == null) {
							filePanel.removeCheckBox(fileNames[i], size);
							//filePanel.setCheckBoxInvalid(fileNames[i], "deleted"); // 
							//model.setAnnotation(new File(fileNames[i]), "deleted" + FileModel.getInvalidMarker());
							// subtract original size (now size is 0)
							//model.increaseAllSize( -size);
						} else {
							if (newError.equals("is folder")) {
								dirToDelete.add(fileNames[i]);
							} else {
								errorMessageWipe += "\n" + newError;
							}
						}
					}
					// manually delete remaining directories if they are empty
					int foldersLen = dirToDelete.size();
					for (int i = foldersLen-1; i >=0; i--) {// begin with last: file hierarchy is from parent folders
						File folder = new File(dirToDelete.get(i));
						// delete only if there are no remaining files
						if (folder.exists() && (FileTools.getNumberAndSize(dirToDelete.get(i))[1] == 0)){
							folder.delete();
							filePanel.setCheckBoxInvalid(dirToDelete.get(i), "deleted");
							model.setAnnotation(folder, "deleted" + FileModel.getInvalidMarker());
						}
					}
					frame.setCursor(CursorManager.getDefaultCursor());
					messageLabel.setText("");
				}
				
				filePanel.updateNumberAndSize();
				if (errorMessageWipe.length() > 0) {
					PeaDialog.showMessage(getFileDisplayPanel(), "Deletion failed: \n" + errorMessageWipe, "Wipe", 0);
				} else {
					PeaDialog.showMessage(getFileDisplayPanel(), FilePEASetting.fileTranslate("file_wiped"), "Wipe", 1);					
				}
			}
			// add zip file to FileTypePanel
			File zipFile = new File(zipFolder.getAbsolutePath() + File.separator + zipName);

			// ad or update file:
			filePanel.getFileComposer().addOrUpdateFile(zipFile, false);		
			filePanel.updateNumberAndSize();
			filePanel.updateFileView();			
			
		} else if ( command.equals("zipFolder") ) {	
			TestLog.o(LockFrameFile.class, "zip folder");			
			// select a file to wipe:
			PeaFileChooser chooser = new PeaFileChooser(this);
			chooser.customizeFileChooser("Zip", PeaProperties.getVmBridge().translate("select_folder_to_zip"), false);
			chooser.setFileSelectionMode( JFileChooser.DIRECTORIES_ONLY);
			//In response to a button click:
			int returnVal = chooser.showOpenDialog(this);
			if (chooser.getSelectedFile() == null || returnVal != JFileChooser.APPROVE_OPTION) {
				TestLog.o(LockFrameFile.class, "No folder selected...");
				return;					
			} else {
				File folder = chooser.getSelectedFile();

				String errorMessage = null;				
				
				if (folder.list().length == 0) {
					errorMessage = PeaProperties.getVmBridge().translate("invalid_empty_directory");
				} else {
					try { //  Wait cursor is also set in ZipStuff
						// set the wait cursor: 
						frame.setCursor(CursorManager.getWaitCursor());
						messageLabel.setForeground(Color.BLACK);
						messageLabel.setText("... " + PeaProperties.getVmBridge().translate("zip") + " " +
								PeaProperties.getVmBridge().translate("in_progress") + "... ");
					} finally {					
						errorMessage = ZipStuff.zipFolder(folder.getAbsolutePath(), folder.getName(), this);			
						frame.setCursor(CursorManager.getDefaultCursor());
						messageLabel.setText("");
					}
				}

				if (errorMessage != null) {
					PeaDialog.showMessage(this, FilePEASetting.fileTranslate("zip_failed") + ": \n" + errorMessage, "Zip", 0);
				} else {
					PeaDialog.showMessage(this, FilePEASetting.fileTranslate("zip_created") + ": \n" + folder.getAbsolutePath() + ".zip"
							+ "\n" + FilePEASetting.fileTranslate("beside_folder"), "Zip", 1); 
					// add or update file:
					filePanel.getFileComposer().addOrUpdateFile(new File(folder.getAbsolutePath() + ".zip"), false);		
					filePanel.updateNumberAndSize();
					filePanel.updateFileView();		
				}
			}
			
		} else if ( command.equals("wipe") ) {	
			
			// select a file to wipe:
			PeaFileChooser chooser = new PeaFileChooser(this);
			chooser.customizeFileChooser("Wipe", PeaProperties.getVmBridge().translate("select_file_to_wipe"), false);
			chooser.setFileSelectionMode( JFileChooser.FILES_ONLY);
			//In response to a button click:
			int returnVal = chooser.showOpenDialog(this);
			if (chooser.getSelectedFile() == null || returnVal != JFileChooser.APPROVE_OPTION) {
				TestLog.o(LockFrameFile.class, "No file wiped and deleted...");
				return;					
			} else {
				File file = chooser.getSelectedFile();
	//			long size = FileTools.getFileSize(file);
				String errorMessage = null;
				try {
					messageLabel.setForeground(Color.BLACK);
					messageLabel.setText("... " + PeaProperties.getVmBridge().translate("wipe_and_delete") + " " +
							PeaProperties.getVmBridge().translate("in_progress") + "... ");
				} finally {
					errorMessage = FileTools.wipeFile(file.getAbsolutePath(), true);
					messageLabel.setText("");					
				}
				if (errorMessage == null) {
					PeaDialog.showMessage(this, FilePEASetting.fileTranslate("file_wiped") + ": \n" + file.getAbsolutePath(), "Wipe", 1);
				} else {
					PeaDialog.showMessage(this, "Deletion failed: \n" + file.getAbsolutePath()
					+"\n" + errorMessage, "Wipe", 0);
					return;
				}
				// check if map contains file, then mark as invalid and subtract allSize manually
				FileModel model = filePanel.getFileComposer().getFileModel();
				if (model.containsFile(file)){
					// remove file from map: 
					filePanel.setCheckBoxInvalid(file.getAbsolutePath(), "deleted");
					model.setAnnotation(file, "deleted" + FileModel.getInvalidMarker());
					// subtract original size (now size is 0) done in setCheckBoxInvalid
					//model.increaseAllSize( -size);
					filePanel.updateNumberAndSize();
					filePanel.updateFileView();
				}
			}			
		} else if ( command.equals("setTimeout") ) {	
			TestLog.v(getClass(), "Set timeout...");
			TimeoutSetter ts = TimeoutSetter.getInstance(this);
			ts.setVisible(true);
		}
	}
	
	
	/**
	 * Handle unselected files, encrypt files, clear
	 * keys, set decrypted for shutdown hook and exit
	 */
	@Override
	public void closeProgram(){
		
		ArrayList<String> usedFileNames = getUsedFileNames();
		if (usedFileNames != null) {
			PeaProperties.setLastUsedFiles(usedFileNames);
		}
		
		// if unselected files remain: warning
		if (cancelClosingProcess() == true) {
			// closing process was canceled because of
			// unencrypted remaining files
			return;
		}		
		// if files are not in path file, ask:
		String[] currFileNames = getCurrentFileNames();
		if (currFileNames != null) {
			HashSet<String> availableSet = Converter.arrayToHashSet(currFileNames);
			if (availableSet != null && availableSet.size() > 0) {
				PathFileManager.addFileNamesToPathFile(availableSet, true);
			}
		}		
		// save properties
		new PropertyHandler().updateAllPropValues();
		
		// 4. check if cloud processes are still running and ask to break them
		if (waitForRunningCloudProcesses() == true) {
			return;
		}
		
		try {
			messageLabel.setForeground(Color.BLACK);
			messageLabel.setText(PeaProperties.getVmBridge().translate("encryption") + " " +
					PeaProperties.getVmBridge().translate("in_progress"));

			// set the wait cursor: 
			frame.setCursor(CursorManager.getWaitCursor());

		} finally {			

			// encrypt the files: 
			String errors = encryptCurrentFiles(null);
			if (errors != null) {
				PeaDialog.showMessage(JREProperties.getMainWindow(), errors, 
						PeaProperties.getVmBridge().translate("error"), 0);
			} else {
				// TODO ? Show error on next start as toast ? 
			}
			// set property  to register interrupted closing 
			// and do not show invalidFiles 
			FilePEASetting.setClosedProperly(true);
			FilePEASetting.setShowInvalidPanel(false);
			new PropertyHandler().updateAllPropValues();
			FileControl.getInstance().disableWarningOnOS();
			//FileControl.getInstance().updatePeaPropValues(null)
			
			// clear the sessionKeys:
			CipherStuff.getSessionKeyCrypt().clearKeys();				

			//  Close all providers
			closeOpenCloudProviders();

			FileControl.setDecrypted(false);

			System.exit(0);
		}
	}
	
	@Override
	public void windowClosing(WindowEvent we) { // by X-Button

		closeProgram();
	}
	@Override
	public void windowActivated(WindowEvent arg0) {}
	@Override
	public void windowClosed(WindowEvent arg0) {}
	@Override
	public void windowDeactivated(WindowEvent arg0) {}
	@Override
	public void windowDeiconified(WindowEvent arg0) {}
	@Override
	public void windowIconified(WindowEvent arg0) {}
	@Override
	public void windowOpened(WindowEvent arg0) {}
	

	/*
	 * Encrypt the selected files
	 */
	protected final String encryptCurrentFiles(byte[] keyMaterial){

		String[] fileNames = filePanel.getValidLocalFileNames(true, false);
		
		if (fileNames == null || fileNames.length == 0){
			TestLog.e(getClass(), "No files to encrypt selected");
			FileControl.setDecrypted(false);
			return null;
		}

		// give informations if there has been errors:
		String errors = null;
		if(filePanel.getFileComposer().getFileModel().getAllSize() > (16 * 1024 * 1024)
				|| filePanel.getFileComposer().getFileModel().getFileNumber() > 64) {
			filePanel.startProgressTask();
		}
		try {
			// set the wait cursor: 
			frame.setCursor(CursorManager.getWaitCursor());
			
			String[] errorMessages = CipherStuff.getCipherMode().encryptFiles(
					fileNames, keyMaterial, true, filePanel.getFileComposer(), null);
			filePanel.closeProgressTask();// close automatically

			for (int i = 0; i < errorMessages.length; i++) {
				if (errorMessages[i] != null) {
					errors +=  errorMessages[i] + " - " + fileNames[i] + "\n";
				}
			}	
		} finally {
			frame.setCursor(CursorManager.getDefaultCursor());
		}

		FileControl.setDecrypted(false);
		TestLog.o(LockFrameFile.class, "file(s) encrypted");
		return errors;
	}
	

	/**
	 * Show a warning message if there are file 
	 * left which will remain unencrypted
	 * 
	 * @return	true, if the closing process should be 
	 * 			canceled because of unencrypted files,
	 * 			false if closing process can go on
	 */
	private final boolean cancelClosingProcess(){
		// TODO directories
		// if unselected files remain: warning
		String[] unselectedFiles = filePanel.getUnselectedValidFileNames();
		if ( unselectedFiles != null ) {
			// create String to display:
			StringBuilder unselectedFilesBuilder = new StringBuilder();
			for (int i = 0; i < unselectedFiles.length; i++) {
				unselectedFilesBuilder.append(unselectedFiles[i]);
				unselectedFilesBuilder.append("\n");
			}
			String unselectedFilesString = new String(unselectedFilesBuilder);

			int result = PeaDialog.showQuestionWithScollPane(this, 
					PeaProperties.getVmBridge().translate("unselected_files_left")
					+ "\n\n" + unselectedFilesString, 
					PeaProperties.getVmBridge().translate("warning"), 
					2, // ok-cancel option
					2);// warning message
			if (result == 0) { //JOptionPane.OK_OPTION
				// unencrypted files were accepted, go on
				return false;
			} else { // cancel, close
				// don't close...
				return true;
			}
		} else {
			// nothing to do, go on
			return false;
		}
	}

	
	//==================================
	// Getter & Setter
	protected final static FileTypePanel getFileDisplayPanel() {
		if (filePanel == null) {
			if(JREProperties.getMainWindow() instanceof PeaLockFrame) {		
				filePanel = PeaControl.getDialog().getEncryptedFileTypePanel();
			} else {
				if (JREProperties.getTypePanel() instanceof FileTypePanel) {
					filePanel = (FileTypePanel) JREProperties.getTypePanel();
				}
			}
		}
		return filePanel;
	}
	protected final static void setFileDisplayPanel(FileTypePanel _fdp) {
		filePanel = _fdp;
	}
	

	@Override
	public String decryptAndLoadFileToView(String newFileName) {
		
		//TestLog.o(LockFrameFile.class, "DecryptAndLoad fileName: " + newFileName);
		// decrypt the file
		String[] fileNames = {newFileName};
		String[] errorMessages = CipherStuff.getCipherMode().decryptFiles( fileNames, 
				CipherStuff.getSessionKeyCrypt().getKey()[0],
				true, filePanel.getFileComposer() );

		// add to view
		if (errorMessages == null || errorMessages[0] == null) {
			filePanel.getFileComposer().addOrUpdateFile(new File(newFileName), false);
			filePanel.updateFileView();
		} else {			
			String extraMsg = "";
			if (errorMessages[0].equals(PeaProperties.getVmBridge().translate("data_integrity_violated"))) {
			extraMsg = "\n" + PeaProperties.getVmBridge().translate("integrity_check_failed_message");
			}
			PeaDialog.showMessage(this, 
					newFileName + "\n" +
					errorMessages[0] + extraMsg, 
					PeaProperties.getVmBridge().translate("error"), 0);
		}
		return null;
	}

	@Override
	public String[] getCurrentFileNames() {
		return filePanel.getValidLocalFileNames(true, true);// selected files, no directories
	}

	@Override
	public void setCurrentFileNames(String[] fileNamesToSet) {
		new UnexpectedValueException("Method", "setCurrentFileNames", "is not implemented").printDescription();
	}

	@Override
	public void setMessage(String string, boolean isError) {
		if (isError == true) {
			messageLabel.setForeground(Color.RED);
		} else {
			messageLabel.setForeground(Color.BLACK);
		}
		messageLabel.setText(string);
		
	}

	@Override
	public boolean checkUnsavedContentChanges() {
		return false;
	}
	@Override
	public void markUnsavedContentChanges() {
		// not relevant		
	}
	@Override
	public void markContentIsSaved() {
		// not relevant		
	}
	@Override
	public boolean isContentUnsaved() {
		return false;
	}
	@Override
	public void updateModelWithCloudFiles(File[] files, String providerName) {
		// there is no model
	}

	@Override
	public String saveCurrentContentInCloudTempFile(String providerName, String fileNameToStore) {
		new UnexpectedValueException("Method", "saveCurrentContent...", "is not implemented").printDescription();
		return null;
	}

	@Override
	public String saveCurrentContent() {
		return null;
	}

	@Override
	public String[] chooseFileNamesToApply() {
		// nothing to do, file names to apply changes are all selected files
		return null;
	}

	@Override
	public boolean changePassword() {

		String[] unselectedFiles = filePanel.getUnselectedValidFileNames();
		if (unselectedFiles == null){
			// nothing to do
		} else {
			if (unselectedFiles.length == 0){
				// nothing to do
			} else {
				// there are unselected valid files:
				// do not allow to change the password or apply changes
				PeaDialog.showMessage(this, FilePEASetting.fileTranslate("remaining_files_for_password_change"));
				return false;
			}
		}
		
		LockFrameFile.frame.storeValuesBeforeChange();
		
		String error = changePasswordOrApplyChanges(this, null, false);
		if (error != null) {
			PeaDialog.showMessage(this, error);
			return false;
		} else {
			TestLog.v(getClass(), "Successfully changed password");
			return true;
		}
	}

	/**
	 * @param timer the timer to set
	 */
	public static void setTimerStarted(boolean _timerStarted) {
		LockFrameFile.timerStarted = _timerStarted;
	}

	/**
	 * Set the timeout time to display
	 * 
	 * @param time	time as String
	 */
	@Override
	public void setTimeoutTime(String time) {
		if (timeoutButton != null) {
			timeoutButton.setText(time);
		}
	}

	@Override
	public ArrayList<String> getUsedFileNames() {

		String[] currentFileNames = getCurrentFileNames();
		if (currentFileNames != null) {
			return Converter.arrayToArrayList(currentFileNames);
		} else {
			return null;
		}
	}
}